#pragma once

#include <memory>
#include <vector>

#include <enyx/cores/result.hpp>
#include <enyx/cores/tcp/tcp.hpp>
#include <enyx/cores/tcp/session.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores_c/tcp/session_monitor.h>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_tcp_session_monitor>
{
    void
    operator()(::enyx_tcp_session_monitor * ptr) const
    {
        ::enyx_tcp_session_monitor_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

/**
 * Session monitor object.
 *
 * This object is used to monitor tcp sessions and detect status changes.
 *
 * Usage:
 * @code{.cpp}
 * enyx::tcp::session_monitor monitor{tcp};
 *
 * auto session = tcp.get_available_session().value();

 * monitor.add_session(session).value();
 *
 * session.connect(address, port, &parameters).value();
 *
 * sleep(5);
 *
 * auto events = monitor.poll_events(events);
 * if (event.size() == 0) {
 *     std::ostringstream error;
 *     error << "Session " << session.id() << "did not open!";
 *     throw std::runtime_error{error.str()};
 * }
 *
 * switch (events[0].new_state) {
 * case ENYX_TCP_SESSION_STATE_OPENING:
 *     std::cout << "Session " << events[0].session_id << " is connecting" << std::endl;
 *     break;
 * case ENYX_TCP_SESSION_STATE_ESTABLISHED:
 *     std::cout << "Session " << events[0].session_id << " is connected" << std::endl;
 *     break;
 * default:
 *     std::cout << "Unexpected status CLOSING for session "
 *               << events[0].session_id << std::endl;
 *     break;
 * };
 *
 * session.close.value();
 *
 * std::sleep(5);
 *
 * auto events = monitor.poll_events(events);
 * if (event.size() == 0) {
 *     std::ostringstream error;
 *     error << "Session " << session.id() << "did not close!";
 *     throw std::runtime_error{error.str()};
 * }
 *
 * switch (events[0].new_state) {
 * case ENYX_TCP_SESSION_STATE_CLOSING:
 *     std::cout << "Session " << events[0].session_id << " is closing" << std::endl;
 *     break;
 * case ENYX_TCP_SESSION_STATE_CLOSED:
 *     std::cout << "Session " << events[0].session_id << " is closed" << std::endl;
 *     break;
 * default:
 *     std::cout << "Unexpected status for session "
 *               << events[0].session_id << std::endl;
 *     break;
 * };
 * @endcode
 */
class session_monitor {
public:
    /**
     * @copydoc enyx_tcp_session_monitor_event
     */
    using event = ::enyx_tcp_session_monitor_event;
public:
    /**
     * Create a session_monitor.
     *
     * @param tcp The tcp this monitor uses
     * @throw system_error on failure
     */
    session_monitor(tcp const & tcp);

    /**
     * @copybrief ::enyx_tcp_session_monitor_add_session
     *
     * @param session The session to monitor
     * @return A result object containing an error on failure
     */
    result<void>
    add_session(session const & session) noexcept;

    /**
     * @copybrief ::enyx_tcp_session_monitor_remove_session
     *
     * @param session The session to stop monitoring
     * @return A result object containing an error on failure
     */
    result<void>
    remove_session(session const & session) noexcept;

    /**
     * Poll all monitored sessions.
     *
     * For each session whose status has changed since last poll, an event will
     * be created.
     *
     * @note This function does a lot of MM access, you may want to wait a bit
     * between each calls.
     *
     * @param monitor The monitor used for polling
     * @return A result object containing events on success or an error on
     *         failure
     */
    result<std::vector<event>>
    poll_events() const noexcept;

    /**
     * @copybrief ::enyx_tcp_session_monitor_count
     *
     * @return The number of sessions
     */
    size_t
    count() const noexcept;

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_tcp_session_monitor *
    handle() const noexcept;

private:
    std::unique_ptr<::enyx_tcp_session_monitor> monitor_;
};

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/tcp/session_monitor.ipp>
