#pragma once

#include <cerrno>
#include <system_error>

#include <enyx/cores_c/tcp/data_stream.h>

#include <enyx/cores/data_stream/sink.hpp>
#include <enyx/cores/data_stream/source.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

/// C++ alias of @b C @ref enyx_tcp_metadata
using metadata = enyx_tcp_metadata;

/**
 * C++ TCP source abstraction
 *
 * Usage:
 * @code
 * auto on_tcp_data = [] (std::uint8_t const * data, std::uint32_t size,
 *                        void const * metadata) {
 *      std::cout << "TCP data received!" << std::endl;
 * };
 *
 * source tcp{on_tcp_data};
 * enyx_hw_source hw_source{stream, tcp};
 *
 * do
 *     hw_source.poll_once().v();
 * while (! is_exit_requested);
 * @endcode
 */
class source final
{
public:
    /**
     *  Constructs a TCP source calling @p handler on data.
     *
     *  @note Handler signature is:
     *      void(std::uint8_t const * data, std::uint32_t size,
     *           void const * metadata);
     */
    template<typename Handler>
    source(Handler && handler);

    /**
     * TCP source callback to invoke when data to be decoded is available.
     *
     * @param data: The TCP data to decode
     * @param size: Data size
     * @param metadata: Metadata associated with passed data
     */
    void
    operator()(std::uint8_t const * data,
               std::uint32_t size,
               void const * metadata);

    /**
     * Get the current source as a C source
     *
     * @return A C source instance
     */
    ::enyx_data_stream_source
    as_next_source() const noexcept;

private:
    data_stream::source_ptr next_source_;
};

/**
 *  @copybrief enyx_tcp_create_sink
 *
 *  @param data_sink The sink of the TCP data
 *  @return the created sink
 */
result<data_stream::sink>
create_sink(data_stream::sink & data_sink);

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/tcp/data_stream.ipp>
