#include "enyx/cores_c/i2c/i2c.h"
ENYX_CORES_NAMESPACE_BEGIN

namespace i2c {

inline
bus::bus(std::shared_ptr<::enyx_i2c> i2c_c, ::enyx_i2c_bus * bus_c)
    : i2c_c_(i2c_c)
    , bus_c_(bus_c)
{
    if (! bus_c_)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to create I2C bus object."};
}

inline result<size_t>
bus::read(const uint8_t address, uint8_t * data, size_t size)
{
    auto res = enyx_i2c_bus_read(handle(), address, data, size);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline result<void>
bus::write(const uint8_t address, uint8_t const * data, size_t size)
{
    if (enyx_i2c_bus_write(handle(), address, data, size) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<size_t>
bus::read_register(const uint8_t address, const uint8_t reg,
                   uint8_t * data, size_t size)
{
    auto res = enyx_i2c_bus_read_register(handle(), address, reg, data, size);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline result<void>
bus::write_register(const uint8_t address, const uint8_t reg,
                    uint8_t const * data, size_t size)
{
    if (enyx_i2c_bus_write_register(handle(), address, reg, data, size) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
bus::set_clock_prescale(enyx_i2c_clock_prescale clock_prescale)
{
    if (enyx_i2c_bus_set_clock_prescale(handle(), clock_prescale) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
bus::set_read_timeout(uint64_t timeout_ms)
{
    if (enyx_i2c_bus_set_read_timeout(handle(), timeout_ms) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline enyx_i2c_bus const *
bus::handle() const noexcept
{
    return bus_c_.get();
}

inline enyx_i2c_bus *
bus::handle() noexcept
{
    return bus_c_.get();
}

inline
i2c::i2c(enyx::hw::core & core)
    :i2c_c_(::enyx_i2c_create(core.handle()), ::enyx_i2c_destroy)
{
    if (! i2c_c_)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to create i2c object."};
}

inline result<size_t>
i2c::get_bus_count()
{
    auto count = enyx_i2c_get_bus_count(handle());
    if (count < 0)
        return std::error_code{errno, std::generic_category()};

    return count;
}

inline result<bus>
i2c::get_bus(uint8_t bus_id)
{
    ::enyx_i2c_bus * bus = enyx_i2c_get_bus(handle(), bus_id);
    if (! bus)
        return std::error_code{errno, std::generic_category()};

    return {i2c_c_, bus};
}

inline enyx_i2c const *
i2c::handle() const noexcept
{
    return i2c_c_.get();
}

inline enyx_i2c *
i2c::handle() noexcept
{
    return i2c_c_.get();
}

} /* namespace i2c */

ENYX_CORES_NAMESPACE_END
