"XML core descriptions parser"""

from collections import OrderedDict
from lxml import etree

from .register import Argument, Config, Done, Exec, Return, Monitored
from .register import PageCount, PageID, PageReady, StrIndex, StrValue

from .variable import Bit, Hex, Int, Enum, Bool
from .variable import IPAddress, MACAddress, Time
from .variable import String, Page, Command
from .variable import PageValues

from .core import Core

from .utils import filter_subtree


def core_from_xml(element: etree.Element):
    """Construct from xml element"""
    return Core(element.get("name"), int(element.get("major")),
                int(element.get("minor")), int(element.get("revision")),
                int(element.get("id")), parse_subtree(element))


def enum_from_xml(element: etree.Element):
    """Construct from xml element"""
    values = map(lambda e: (e.get('name'), int(e.get('value'))),
                 element.xpath('./HwValueEnum'))
    values = OrderedDict(sorted(values, key=lambda v: v[1]))
    return Enum(build_register(element), values)


def string_from_xml(element: etree.Element):
    """Construct from xml element"""
    index = hw_value_from_xml(element)
    value = hw_value_from_xml(element.xpath('./HwValue[@class="VAL"]')[0])
    return String.from_registers(index, value)


def page_from_xml(element: etree.Element):
    """Construct from xml element"""
    subtree = parse_subtree(list(element))
    count = next(filter_subtree(subtree, PageCount))
    page_id = next(filter_subtree(subtree, PageID))
    ready = next(filter_subtree(subtree, PageReady), None)
    variables = list(
        filter_subtree(subtree, (PageCount, PageID, PageReady), True))
    values = list(filter(lambda val: isinstance(val, PageValues), subtree))
    return Page(element.get('name'), count, page_id, ready, variables, values)


def command_from_xml(element: etree.Element):
    """Construct from xml element"""
    execute = hw_value_from_xml(element)
    subtree = parse_subtree(list(element))
    args = list(filter_subtree(subtree, Argument))
    done = next(filter_subtree(subtree, Done), None)
    ret = list(filter_subtree(subtree, Return))
    return Command(element.get('name'), execute, args, ret, done,
                   element.get('comment', ''))


def hw_value_from_xml(element: etree.Element):
    """Construct any variable representing an Hardware Value"""
    classes = {
        'bit': Bit,
        'bool': Bool,
        'enum': Enum,
        'hex': Hex,
        'int': Int,
        'ip': IPAddress,
        'mac': MACAddress,
        'time': Time,
        # registers with type `str` are just the Value register for a String
        # and should be interpreted as Int
        'str': Int,
    }
    # For registers with no type (string_id or cmd_exec) we assume an int type
    cls = classes.get(element.get('type'), Int)
    if cls is Enum:
        return enum_from_xml(element)
    return cls(build_register(element))


def page_values_from_xml(element: etree.Element):
    """Retrieve page values"""
    subtree = parse_subtree(list(element))
    return PageValues({'page_id': element.get('id'), 'content': subtree})


def page_value_from_xml(element: etree.Element):
    """Retrieve page value"""
    return {'name': element.get('name').lower(), 'value': element.get('value')}


def build_register(element: etree.Element):
    """Create register specific class from xml element"""
    classes = {
        'ARG': Argument,
        'CFG': Config,
        'DON': Done,
        'RET': Return,
        'MON': Monitored,
        'PAGE_CNT': PageCount,
        'PAGE_ID': PageID,
        'PAGE_RDY': PageReady,
        'VAL': StrValue,
    }
    class_attr = element.get('class')

    if class_attr in classes:
        cls = classes[class_attr]
    elif element.tag == 'HwCommand':
        cls = Exec
    elif element.tag == 'HwStr':
        cls = StrIndex

    return cls(element.get('name'),
               int(element.get('size')),
               int(element.get('regnum')),
               int(element.get('offset')),
               element.get('comment', ''),
               element.get('value'))


def from_xml(element: etree.Element):
    """Construct any variable from xml element"""
    builders = {
        'HwValue': hw_value_from_xml,
        'HwCommand': command_from_xml,
        'HwPage': page_from_xml,
        'HwStr': string_from_xml,
        'Core': core_from_xml,
        'HwPageValues': page_values_from_xml,
        'HwPageValue': page_value_from_xml,
    }

    if element.tag not in builders:
        raise RuntimeError("Unsupported XML element: {}".format(element.tag))

    return builders[element.tag](element)


def parse_subtree(core: etree.Element):
    """Parse children of xml element into variables (or pages/commands)"""
    return list(map(from_xml, core))
