"""Register types"""


class Register:
    """Represent a hardware register
    """
    def __init__(self, name: str, size: int, regnum: int, offset: int,
                 comment: str, value: str = None):
        # pylint: disable=too-many-arguments
        self.name_ = name
        self.size_ = size
        self.regnum = regnum
        self.offset = offset
        self.comment = comment
        self.value = value

    @property
    def name(self):
        """name property"""
        return self.name_.lower()

    @property
    def byte_offset(self):
        """Amount of bytes in the offset from core address"""
        return self.regnum * 4 + self.offset // 8

    @property
    def bit_offset(self):
        """Amount of bit in the offset from byte offset"""
        return self.offset % 8

    @property
    def size(self):
        """size in bit of register"""
        return self.size_

    @property
    def type(self):
        """type property"""
        return type(self).__name__


class Argument(Register):
    """RW Only in Command"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return True


class Config(Register):
    """RW"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return True


class Done(Register):
    """RO only in command (optional)"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False


class Exec(Register):
    """
        This class does not exist in the xml but exists in the core themselves.
    """
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return True


class Return(Register):
    """RO only in Command"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False


class Monitored(Register):
    """RO"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False


class PageCount(Register):
    """RO only in Page"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False


class PageID(Register):
    """RW only in page"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return True


class PageReady(Register):
    """RO only in page"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False


class StrIndex(Register):
    """
        This class does not exist in the xml but exists in the core themselves.
    """
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return True


class StrValue(Register):
    """RO only in string"""
    @property
    def read(self):
        """Readability property"""
        return True

    @property
    def write(self):
        """Writability property"""
        return False
