/*
 * HFP linux module entry point.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/msi.h>
#include <linux/fs.h>
#include <linux/sysfs.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/pci_regs.h>
#include <linux/delay.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_char_device.h"
#include "enyx_hfp_irq_list.h"
#include "enyx_hfp_bus.h"

#define HFP_IRQ_MAX_COUNT 32

struct pci_drvdata {
    struct enyx_hfp_irq_list * irq_list;
    struct enyx_hfp_bus * bus;
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 8, 0)

static int
allocate_all_interrupts(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);
    int irq_count;

    irq_count = pci_alloc_irq_vectors(pdev, 1, HFP_IRQ_MAX_COUNT,
                                      PCI_IRQ_MSI | PCI_IRQ_MSIX);

    if (irq_count <= 0)
        goto allocate_interrupts_failed;

    while (irq_count--)
        if (enyx_hfp_irq_list_put(drvdata->irq_list,
                             pci_irq_vector(pdev, irq_count),
                             irq_count) < 0) {
            goto enyx_hfp_irq_list_put_failed;
        }

    return 0;

enyx_hfp_irq_list_put_failed:
allocate_interrupts_failed:
    return -1;
}

#else

static int
allocate_msix_interrupts(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);
    struct msix_entry entries[HFP_IRQ_MAX_COUNT];
    int irq_count;
    int err;

    for (irq_count = 0; irq_count != HFP_IRQ_MAX_COUNT; ++irq_count)
         entries[irq_count].entry = irq_count;

    err = irq_count = pci_enable_msix(pdev, entries, HFP_IRQ_MAX_COUNT);
    if (irq_count == 0)
        irq_count = HFP_IRQ_MAX_COUNT;
    else if (irq_count > 0)
        err = pci_enable_msix(pdev, entries, irq_count);

    if (err)
        goto pci_enable_msix_failed;

    dev_dbg(&pdev->dev, "Allocated %d MSI-X interrupts\n", irq_count);

    while (irq_count--)
        if (enyx_hfp_irq_list_put(drvdata->irq_list, entries[irq_count].vector,
                                                entries[irq_count].entry) < 0) {
            err = -ENODEV;
            goto enyx_hfp_irq_list_put_failed;
        }

    return 0;

enyx_hfp_irq_list_put_failed:
    pci_disable_msix(pdev);
pci_enable_msix_failed:
    return err;
}

static int
allocate_msi_interrupts(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);
    int irq_count;
    int err;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 12, 0)
    err = irq_count = pci_enable_msi_range(pdev, 1, HFP_IRQ_MAX_COUNT);
#else
    err = irq_count = pci_enable_msi_block(pdev, HFP_IRQ_MAX_COUNT);

    if (irq_count > 0)
        err = pci_enable_msi_block(pdev, irq_count);
    else
        irq_count = HFP_IRQ_MAX_COUNT;
#endif
    if (err < 0)
        goto pci_enable_msi_block_failed;

    dev_dbg(&pdev->dev, "Allocated %d MSI interrupts\n", irq_count);

    while (irq_count--)
        if (enyx_hfp_irq_list_put(drvdata->irq_list,
                             pdev->irq + irq_count,
                             irq_count) < 0) {
            err = -ENODEV;
            goto enyx_hfp_irq_list_put_failed;
        }

    return 0;

enyx_hfp_irq_list_put_failed:
    pci_disable_msi(pdev);
pci_enable_msi_block_failed:
    return err;
}

#endif

static int
allocate_interrupts(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);

    drvdata->irq_list = enyx_hfp_irq_list_create(HFP_IRQ_MAX_COUNT);
    if (! drvdata->irq_list)
        goto enyx_hfp_irq_list_create_failed;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 8, 0)
    if (allocate_all_interrupts(pdev)) {
#else
    if (allocate_msix_interrupts(pdev) && allocate_msi_interrupts(pdev)) {
#endif
        dev_err(&pdev->dev, "Can't request MSI/MSI-X irqs\n");
        goto allocate_interrupts_failed;
    }

    return 0;

allocate_interrupts_failed:
    enyx_hfp_irq_list_destroy(drvdata->irq_list);
enyx_hfp_irq_list_create_failed:
    return -1;
}

static void
free_interrupts(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);

    pci_disable_msix(pdev);
    pci_disable_msi(pdev);

    enyx_hfp_irq_list_destroy(drvdata->irq_list);
}

static int
enyx_hfp_pci_create_devices(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);
    int err;

    err = allocate_interrupts(pdev);
    if (err < 0)
        goto allocate_interrupts_failed;

    drvdata->bus = create_enyx_hfp_bus(pci_resource_start(pdev, 0),
                                  drvdata->irq_list,
                                  &pdev->dev);

    if (drvdata->bus == NULL) {
        err = -EIO;
        goto err_create_enyx_hfp_bus;
    }

    err = scan_enyx_hfp_bus(drvdata->bus);
    if (err < 0)
        goto err_scan_enyx_hfp_bus;

    return 0;

err_scan_enyx_hfp_bus:
    destroy_enyx_hfp_bus(drvdata->bus);
err_create_enyx_hfp_bus:
    free_interrupts(pdev);
allocate_interrupts_failed:
    return err;
}

static void
enyx_hfp_pci_destroy_devices(struct pci_dev * pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);

    /* Loop until the lock has been released */
    while (clear_enyx_hfp_bus(drvdata->bus) < 0)
        dev_err(&pdev->dev, "Retrying bus destruction\n");

    destroy_enyx_hfp_bus(drvdata->bus);

    free_interrupts(pdev);
}

static int
reload(struct pci_dev * pci_dev, unsigned reload_duration)
{
    struct pci_dev * bridge_dev = pci_dev->bus->self;
    int err;
    u16 link_control;
    u8 cap_exp;

    /* Disable Endpoint->CPU TLP forwarding on bridge*/
    pci_clear_master(bridge_dev);

    err = pci_save_state(pci_dev);
    if (err < 0)
        goto pci_save_state_failed;

    /* Retrieve PCI Express capability offset */
    cap_exp = pci_find_capability(bridge_dev, PCI_CAP_ID_EXP);

    /* Retrieve the actual value of the link control register */
    pci_read_config_word(bridge_dev, cap_exp + PCI_EXP_LNKCTL, &link_control);

    /* Disable link */
    pci_write_config_word(bridge_dev, cap_exp + PCI_EXP_LNKCTL,
                          link_control | PCI_EXP_LNKCTL_LD);

    msleep(1);

    /* Reenable link */
    pci_write_config_word(bridge_dev, cap_exp + PCI_EXP_LNKCTL, link_control);

    msleep(1);

    /* Retrain link */
    pci_write_config_word(bridge_dev, cap_exp + PCI_EXP_LNKCTL,
                          link_control | PCI_EXP_LNKCTL_RL);

    msleep(reload_duration);

    /* Restore the PCI state as the firmware reload has reset everything */
    pci_restore_state(pci_dev);

    /* Enable Endpoint->CPU TLP forwarding on bridge */
    pci_set_master(pci_dev->bus->self);

pci_save_state_failed:
    return err;
}

static ssize_t
on_reload_fpga(struct device *dev,
               struct device_attribute *attr,
               const char *buf, size_t count)
{
    struct pci_dev * pci_dev = container_of(dev, struct pci_dev, dev);
    unsigned reload_duration;
    int err;

    if (count >= (PAGE_SIZE - 1))
        return -EINVAL;

    if (sscanf(buf, "%u", &reload_duration) != 1)
        return -EINVAL;

    if (reload_duration == 0)
        return -EINVAL;

    err = reload(pci_dev, reload_duration);
    if (err < 0)
        return err;

    return count;
}

static DEVICE_ATTR(reload_fpga, 0200, NULL, on_reload_fpga);

static int
enyx_hfp_pci_probe(struct pci_dev *pdev,
              const struct pci_device_id *id)
{
    struct pci_drvdata * drvdata;
    int err;

    dev_dbg(&pdev->dev, "Probing\n");

    err = pci_enable_device(pdev);
    if (err < 0) {
        dev_err(&pdev->dev, "Can't enable PCI device\n");
        goto err_pci_enable;
    }

    err = pci_request_regions(pdev, MODULE_NAME);
    if (err < 0) {
        dev_err(&pdev->dev, "Can't get memory/IO regions\n");
        goto err_pci_request_regions;
    }

    err = pci_set_consistent_dma_mask(pdev, DMA_BIT_MASK(64));
    if (err < 0) {
        dev_err(&pdev->dev, "Can't set DMA mask to 64bit\n");
        goto err_set_dma_mask;
    }

    pci_set_master(pdev);

    drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
    if (! drvdata) {
        err = -ENOMEM;
        dev_err(&pdev->dev, "Can't allocate pci struct\n");
        goto err_pci_drvdata_alloc;
    }
    pci_set_drvdata(pdev, drvdata);

    err = device_create_file(&pdev->dev, &dev_attr_reload_fpga);
    if (err < 0)
        goto err_device_create_file;

    err = enyx_hfp_pci_create_devices(pdev);
    if (err < 0)
        goto err_enyx_hfp_pci_create_devices;

    return 0;

err_enyx_hfp_pci_create_devices:
    device_remove_file(&pdev->dev, &dev_attr_reload_fpga);
err_device_create_file:
    kfree(drvdata);
err_pci_drvdata_alloc:
    pci_clear_master(pdev);
err_set_dma_mask:
    pci_release_regions(pdev);
err_pci_request_regions:
    pci_disable_device(pdev);
err_pci_enable:
    return err;
}

static void
enyx_hfp_pci_remove(struct pci_dev *pdev)
{
    struct pci_drvdata * drvdata = pci_get_drvdata(pdev);

    dev_dbg(&pdev->dev, "Removing\n");

    enyx_hfp_pci_destroy_devices(pdev);

    device_remove_file(&pdev->dev, &dev_attr_reload_fpga);

    kfree(drvdata);

    pci_clear_master(pdev);
    pci_release_regions(pdev);
    pci_disable_device(pdev);
}

static pci_ers_result_t
enyx_hfp_pci_error_detected(struct pci_dev *dev, pci_channel_state_t state)
{
    pci_disable_device(dev);

    return PCI_ERS_RESULT_DISCONNECT;
}

static pci_ers_result_t
enyx_hfp_pci_mmio_enabled(struct pci_dev *dev)
{
    return PCI_ERS_RESULT_CAN_RECOVER;
}

static pci_ers_result_t
enyx_hfp_pci_slot_reset(struct pci_dev *dev)
{
    return PCI_ERS_RESULT_DISCONNECT;
}

static void
enyx_hfp_pci_resume(struct pci_dev *dev)
{
}

#define ENYX_PCIE_VENDOR_ID (0x1d8f)
#define ENYX_PCIE_DEV_ID_UNDEFINED (0x1)
#define ENYX_PCIE_DEV_ID_FPB1 (0x2)
#define ENYX_PCIE_DEV_ID_FPB2 (0x3)
#define QEMU_PCIE_VENDOR_ID (0x1234)
#define QEMU_PCIE_DEV_ID_MOCK (0x1)

static const struct pci_device_id enyx_hfp_pci_device_table[] = {
    {PCI_DEVICE(ENYX_PCIE_VENDOR_ID, PCI_ANY_ID)},
    {0},
};
MODULE_DEVICE_TABLE(pci, enyx_hfp_pci_device_table);

static struct pci_error_handlers enyx_hfp_pci_error_handlers = {
    .error_detected = enyx_hfp_pci_error_detected,
    .mmio_enabled = enyx_hfp_pci_mmio_enabled,
    .slot_reset = enyx_hfp_pci_slot_reset,
    .resume = enyx_hfp_pci_resume,
};

static struct pci_driver enyx_hfp_pci_driver = {
    .name = THIS_MODULE->name,
    .id_table = enyx_hfp_pci_device_table,
    .probe = enyx_hfp_pci_probe,
    .remove = enyx_hfp_pci_remove,
    .err_handler = &enyx_hfp_pci_error_handlers,
};

static int __init
enyx_hfp_pci_init_module(void)
{
    int err;

    pr_info("%s %s <support@enyx.com>\n",
            MODULE_NAME, ENYX_HFP_MODULES_VERSION);

    err = pci_register_driver(&enyx_hfp_pci_driver);
    if (err < 0) {
        pr_err("%s: Can't register pci driver\n", MODULE_NAME);
        goto err_pci_register_driver;
    }

    return 0;

err_pci_register_driver:
    return err;
}

module_init(enyx_hfp_pci_init_module);

static void __exit
enyx_hfp_pci_exit_module(void)
{
	pci_unregister_driver(&enyx_hfp_pci_driver);

    pr_debug("%s Exited\n", MODULE_NAME);
}

module_exit(enyx_hfp_pci_exit_module);

MODULE_AUTHOR("David Keller <david.keller@enyx.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Hfp pci module");
MODULE_VERSION(ENYX_HFP_MODULES_VERSION);

