/*
 * HFP id implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_id_usr.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/device.h>

#include "enyx_hfp_common.h"

static DECLARE_BITMAP(device_used, 128);

/* A 32-bit r/o register: page id */
#define HFP_ID_DEVICE_ID                    HFP_REG_ADDR(5, 0)

/* A 8-bit r/o register: page valid */
#define HFP_ID_DEVICE_READY                 HFP_REG_ADDR(6, 0)

/* A 48-bit r/o register: MAC address */
#define HFP_ID_DEVICE_MAC_ADDRESS           HFP_REG_ADDR(7, 0)

struct enyx_hfp_id_usr {
    unsigned int index;

    struct enyx_hfp_device * parent;

    struct device device;

    size_t size;
    uint8_t value[];
};

#define to_enyx_hfp_id_usr(n) \
    container_of(n, struct enyx_hfp_id_usr, device)

static struct class id_class = {
    .name = "enyx_hfp_id",
    .owner = THIS_MODULE,
};

int
enyx_hfp_id_class_register(void)
{
    int err;

    err = class_register(&id_class);
    if (err)
        goto class_register_failed;

    return 0;

class_register_failed:
    return err;
}
EXPORT_SYMBOL(enyx_hfp_id_class_register);

void
enyx_hfp_id_class_unregister(void)
{
    class_unregister(&id_class);
}
EXPORT_SYMBOL(enyx_hfp_id_class_unregister);


static ssize_t
id_show(struct device * device,
        struct device_attribute * attr,
        char * buf)
{
    struct enyx_hfp_id_usr * dev = to_enyx_hfp_id_usr(device);
    return scnprintf(buf, PAGE_SIZE, "MAC=%pM\n", dev->value);
}

static struct device_attribute id_attr_id = __ATTR_RO(id);

static struct attribute * id_attrs[] = {
    &id_attr_id.attr,
    NULL,
};

static const struct attribute_group id_group = {
    .attrs = id_attrs,
};

static const struct attribute_group * id_groups[] = {
    &id_group,
    NULL
};

static void
release_id_usrice(struct device * device)
{
    struct enyx_hfp_id_usr * dev = to_enyx_hfp_id_usr(device);

    clear_bit(dev->index, device_used);

    kfree(dev);
}

static const struct device_type id_type = {
    .name = "enyx_hfp_id",
    .groups = id_groups,
    .release = release_id_usrice,
};

struct enyx_hfp_id_usr *
enyx_hfp_id_usr_create(struct enyx_hfp_device * parent)
{
    struct enyx_hfp_id_usr * dev;
    uint64_t mac_addr;

    if (parent->size < 8 * 4) {
        dev_err(&parent->device,
                "Reported device memory range smaller than expected\n");
        goto size_check_failed;
    }

    dev = kzalloc(sizeof(*dev) + sizeof(mac_addr), GFP_KERNEL);
    if (! dev) {
        dev_err(&parent->device, "Can't allocate id\n");
        goto kzalloc_failed;
    }

    dev->index = parent->bus->index;
    if (test_bit(dev->index, device_used)) {
        dev_err(&parent->device,
                "An id is already registered on this bus\n");
        goto test_bit_failed;
    }
    set_bit(dev->index, device_used);

    dev->parent = parent;
    dev->size = sizeof(mac_addr);

    if (down_interruptible(&dev->parent->lock))
        goto page_select_lock_failed;

    enyx_hfp_write32(dev->parent, HFP_ID_DEVICE_ID, 0);
    if (enyx_hfp_wait_until_val_is(dev->parent, HFP_ID_DEVICE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select page\n");
        goto page_select_failed;
    }

    mac_addr = enyx_hfp_read64(dev->parent, HFP_ID_DEVICE_MAC_ADDRESS);

    up(&dev->parent->lock);

    dev->value[0] = mac_addr >> 40;
    dev->value[1] = mac_addr >> 32;
    dev->value[2] = mac_addr >> 24;
    dev->value[3] = mac_addr >> 16;
    dev->value[4] = mac_addr >> 8;
    dev->value[5] = mac_addr;

    dev->device.parent = &parent->device;
    dev->device.type = &id_type;
    dev->device.class = &id_class;
    if (dev_set_name(&dev->device, "enyx_hfp_id%d", dev->index) < 0) {
        dev_err(&parent->device,
                "Can't set enyx_hfp_id%d device name\n", dev->index);
        goto dev_set_name_failed;
    }

    if (device_register(&dev->device) < 0) {
        dev_err(&parent->device,
                "Can't register enyx_hfp_id%d device\n", dev->index);
        goto device_register_failed;
    }

    dev_info(&dev->device, "Created\n");

    return dev;

device_register_failed:
    put_device(&dev->device);
    /* The device will be garbage collected */
    return NULL;
dev_set_name_failed:
page_select_failed:
page_select_lock_failed:
    clear_bit(dev->index, device_used);
test_bit_failed:
    kfree(dev);
kzalloc_failed:
size_check_failed:
    return NULL;
}
EXPORT_SYMBOL(enyx_hfp_id_usr_create);

void
enyx_hfp_id_usr_destroy(struct enyx_hfp_id_usr * dev)
{
    device_unregister(&dev->device);
}
EXPORT_SYMBOL(enyx_hfp_id_usr_destroy);


