#pragma once

#include <cstdint>
#include <functional>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/c2a_stream.hpp>

#include <enyx/hw_c/mocking/c2a_stream.h>

ENYX_HW_NAMESPACE_BEGIN
namespace mocking {

/**
 * Represent a mocked C2A stream interface.
 *
 * In order to customize it, please inherit and register it using the mocked
 * accelerator class.
 */
class c2a_stream_interface
{
public:
    c2a_stream_interface();
    /**
     * Destructor.
     */
    virtual ~c2a_stream_interface() = default;

    /**
     * Called when the stream is created.
     *
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int()> on_create;

    /**
     * Called when the stream is destroyed.
     */
    std::function<void()> on_destroy;

    /**
     * Called to get the uid.
     *
     * @return the uid
     */
    std::function<std::string()> get_uid;

    /**
     * Called to get the name.
     *
     * @return the name
     */
    std::function<std::string()> get_name;

    /**
     * Called to get the mtu.
     *
     * @return the mtu
     */
    std::function<std::uint32_t()> get_mtu;

    /**
     * Called to get the size.
     *
     * @return the size
     */
    std::function<std::size_t()> get_size;

    /**
     * Called to send @p size bytes of @p data over the stream.
     *
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(void const *data, uint32_t size)> send;

    /**
     * Called to send @p vectors_count data @p vectors over the stream.
     *
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(enyx_hw_c2a_stream_io_vec const *vectors,
                      uint32_t vectors_count)> send_io_vec;

    /**
     * Called to check if the stream  is ready.
     *
     * @return true if the stream is ready, false otherwise.
     */
    std::function<bool()> is_ready;

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is idling, false otherwise.
     */
    std::function<bool()> is_idle;

    /**
     * Called to wait until the stream is idling (i.e. it is not currently
     * processing any data).
     *
     * @note This function should return -1 and set @b errno if the
     *       stream is still not idling after @p us_timeout.
     *
     * @param us_timeout The duration of the wait.
     * @return 0 if the stream is idling, -1 otherwise.
     */
    std::function<int(uint32_t us_timeout)> wait_until_idle;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_c2a_stream_interface *
    handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_c2a_stream_interface const *
    handle() const noexcept;

private:
    std::unique_ptr<::enyx_hwm_c2a_stream_interface> handle_;
};

using basic_c2a_stream ENYX_HW_CXX_DEPRECATED(
            "Replaced by c2a_stream_interface") = c2a_stream_interface;

} /* namespace mocking */
ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/c2a_stream.ipp>
