/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  cores description.
 */
#pragma once

#include <cerrno>
#include <system_error>
#include <enyx/hw/result.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/register_description.hpp>
#include <enyx/hw_c/core_description.h>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_core_description
 */
class core_description
{
public:
    /**
     * Represents a register and its associated name
     *
     * @since 5.0.0
     */
    struct named_register
    {
        /**
         * Construct a named register.
         *
         * @param name The name of the register.
         * @param description Associated description.
         *
         * @since 5.0.0
         */
        named_register(std::string const & name,
                       register_description const & description);
        /**
         * The register name
         *
         * @since 5.0.0
         */
        std::string name;

        /**
         * The register description
         *
         * @since 5.0.0
         */
        register_description description;
    };

    /// Represents a list of named registers
    using named_registers = std::vector<named_register>;

    /// Represents the core name
    using core_name = std::string;

public:
    /**
     * Construct a core description.
     *
     * @param name The new core_description name
     * @param descriptor The new core_description descriptor.
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    core_description(core_name const & name,
                     core::descriptor_type const & descriptor);

    /**
     * Construct a core description from a C core_description.
     *
     * @param description The C core description.
     *
     * @since 5.0.0
     */
    core_description(std::shared_ptr<enyx_hw_core_description> description);

    /**
     * Get the name of the core associated with this description.
     *
     * @return The core name
     *
     * @since 5.0.0
     */
    core_name
    get_name() const noexcept;

    /**
     * Get the core descriptor associated with this description.
     *
     * @return The associated core descriptor
     *
     * @since 5.0.0
     */
    core::descriptor_type
    get_associated_core_descriptor() const noexcept;

    /**
     * Add a register description to the core description.
     *
     * @param name Name of the register.
     * @param reg Corresponding description.
     * @return A result object containing an error on failure.
     *
     * @since 5.0.0
     */
    result<void>
    add_register_description(std::string const & name,
                             register_description const & reg) noexcept;

    /**
     * Find a register matching @p name.
     *
     * @param name The register name.
     * @returns A result object containing either the description or an error.
     *
     * @since 5.0.0
     */
    result<register_description>
    find_register(std::string const& name) const noexcept;

    /**
     * Retrieve the complete list of register associated
     * with this core_description.
     *
     * @returns The list of registers
     *
     * @since 5.0.0
     */
    named_registers
    get_all_registers() const noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_description object.
     *
     * @return The C enyx_hw_core_description object.
     *
     * @since 5.0.0
     */
    enyx_hw_core_description *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_description object.
     *
     * @return The C enyx_hw_core_description object.
     *
     * @since 5.0.0
     */
    enyx_hw_core_description const *
    handle() const noexcept;

private:
    std::shared_ptr<enyx_hw_core_description> description_;
};

/**
 * Claim ownership and make a shared_ptr of C core_description from a raw C
 * @p description pointer.
 *
 * @param description The C raw pointer to use.
 * @return A shared ptr of @p description
 * @throws std::system_error if description was nullptr.
 *
 * @since 5.0.0
 */
std::shared_ptr<enyx_hw_core_description>
make_core_description_ptr(enyx_hw_core_description * description);

ENYX_HW_NAMESPACE_END

#include <enyx/hw/core_description.ipp>
