/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator device.
 */
#pragma once

#include <enyx/hw_c/accelerator.h>

#include <cerrno>
#include <cstdint>
#include <system_error>
#include <vector>
#include <memory>

#include <enyx/hw/fwd.hpp>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/properties.hpp>
#include <enyx/hw/product.hpp>

#include <enyx/hw_c/accelerator.h>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_accelerator_descriptor
 */
class accelerator_descriptor final
{
public:
    /**
     * Retrieve the accelerator's properties
     *
     * @return The accelerator associated properties.
     *
     * @since 5.0.0
     */
    const properties
    get_properties() const;

    /**
     * Construct a descirptor from a C descriptor.
     *
     * @param descriptor The C descriptor to wrap.
     * @throws std::system_error if descriptor is nullptr.
     *
     * @since 5.0.0
     */
    accelerator_descriptor(enyx_hw_accelerator_descriptor const *descriptor);

    /**
     * Retrieve the accelerator's version.
     *
     * @return A result object containing the accelerator version on success
     * or an error on failure
     *
     * @since 5.12.0
     */
    result<product_version>
    get_version() const noexcept;

    /**
     * Retrieve the accelerator's user count.
     *
     * It can be used to check if the accelerator is used before resetting it.
     *
     * @return A result object containing the accelerator user count on success
     * or an error on failure
     *
     * @since 5.12.0
     */
    result<std::size_t>
    count_users() const noexcept;

    /**
     * Reset the accelerator.
     *
     * The accelerator must not be in use to trigger a reset. Resetting it will
     * make the FPGA to reload its firmware from the flash memory.
     *
     * @return A result object containing an error on failure
     *
     * @since 5.12.0
     */
    result<void>
    reset() const noexcept;

    /**
     * Direct access to the underlying C accelerator_descriptor object.
     *
     * @return The C accelerator_descriptor object.
     *
     * @since 5.0.0
     */
    enyx_hw_accelerator_descriptor const *
    handle() const noexcept;

private:
    std::shared_ptr<enyx_hw_accelerator_descriptor const> descriptor_;
};

/**
 * Retrieve all accelerator regardless their properties
 *
 * @return All the accelerators found on this system.
 *
 * @since 5.0.0
 */
std::vector<accelerator_descriptor>
enumerate_accelerators();

/**
 * Retrieve all accelerator matching @p filter argument
 *
 * This function is used to request particular accelerator(s).
 * The @p filter argument should be filled with properties
 * unique to the requested accelerator(s).
 *
 * @param filter accelerator filter
 * @return All the accelerators with equivalent @p filter
 *         found on this system.
 *
 * @since 5.0.0
 */
std::vector<accelerator_descriptor>
enumerate_accelerators(filter const& filter);

ENYX_HW_NAMESPACE_END

#include <enyx/hw/accelerator_descriptor.ipp>
