#include <cstdint>
#include <string>
#include <iostream>
#include <boost/program_options.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/types.hpp>

namespace po = boost::program_options;

ENYX_CORES_NAMESPACE_BEGIN

struct ApplicationConfiguration
{
    std::string accelerator;
    std::uint32_t stack_index;
    std::uint32_t server_interface_id;
    std::uint32_t client_interface_id;
    std::uint64_t payload_size;
    std::uint64_t packet_count;
    double cpu_freq;
    std::uint16_t cpu_core;
    std::string rx_channel_name;
    std::string tx_channel_name;
    std::uint16_t port;
};

inline ApplicationConfiguration
parse_arguments(int argc, char ** argv)
{
    ApplicationConfiguration config;
    po::options_description cmd_desc{"COMMAND LINE OPTIONS"};
    cmd_desc.add_options()
        ("accelerator-index,a",
            po::value<std::string>(&config.accelerator)
                ->default_value("0"),
            "0-index or name of the accelerator to use\n")
        ("stack-index,s",
            po::value<std::uint32_t>(&config.stack_index)
                ->default_value(0),
            "0-index of the stack to use\n")
        ("server-interface-id,S",
            po::value<std::uint32_t>(&config.server_interface_id)
                ->default_value(0),
            "The virtual interface to use for the server session\n")
        ("client-interface-id,C",
            po::value<std::uint32_t>(&config.client_interface_id)
                ->default_value(1),
            "The virtual interface to use for the client session\n")
        ("payload-size,P",
            po::value<std::uint64_t>(&config.payload_size)->required(),
            "Amount of data to send per packets in Bytes\n")
        ("packet-count,n",
            po::value<std::uint64_t>(&config.packet_count)->required(),
            "Number of packets to send\n")
        ("cpu-frequency,f",
            po::value<double>(&config.cpu_freq)->required(),
            "CPU frequency in GHz. This will affect the latency computation, "
            "so make sure to have set this to the right value. Check 'lscpu' "
            "to set this variable.\n")
        ("core-binding,c",
            po::value<std::uint16_t>(&config.cpu_core)->required(),
            "CPU core this process will be bound to. "
            "On multi socket motherboards, in order to have the lowest "
            "latency possible, \n"
            "make sure the CORE_BINDING target CPU is on the "
            "socket that contains the root complex directly "
            "linked to the FPGA board.\n"
            "Specifically, make sure that the FPGA board and the "
            "CPU are in the same NUMA node. "
            "To check this, you can use the following commands :\n"
            "lspci -vv -d 1d8f: | grep NUMA\n"
            "lscpu | grep NUMA\n")
        ("a2c-channel,r",
            po::value<std::string>(&config.rx_channel_name)->required(),
            "Name of the accelerator to CPU channel to use for input "
            "packets\n")
        ("c2a-channel,t",
            po::value<std::string>(&config.tx_channel_name)->required(),
            "Name of the CPU to accelerator channel to use for output "
            "packets\n")
        ("port,p",
             po::value<std::uint16_t>(&config.port)->required(),
             "Port to listen to\n")
        ("help,h",
            "Print the command lines arguments\n")
    ;

    po::variables_map args;
    po::store(po::parse_command_line(argc, argv, cmd_desc), args);
    po::notify(args);

    if (args.count("help"))
    {
        std::cout << cmd_desc << std::endl;
        std::exit(0);
    }

    if (config.payload_size < 2)
    {
        throw std::runtime_error{"--payload-size should be at least 2"};
    }

    return config;
}

ENYX_CORES_NAMESPACE_END
