#include "InterfaceScope.hpp"

#include <cerrno>

#include <future>
#include <sstream>
#include <stdexcept>
#include <vector>

#include <enyx/hw/core.hpp>
#include <enyx/hw/error.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/net_interface/net_interface.hpp>
#include <enyx/cores/hardware_ids.hpp>

#include "Application.hpp"
#include "IpScope.hpp"
#include "LinkScope.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace {

namespace ni = net_interface;

ni::net_interfaces
create_interfaces(hw::core & root,
                  Application::StackType stack_type,
                  std::uint32_t stack_index)
{
    std::vector<ni::net_interfaces> net_interfaces;

    for (auto & core : root)
    {
        auto const hardware_id = core.get_descriptor().hardware_id;
        if (hardware_id == enyx::hardware_ids::TCP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_STD)
                net_interfaces.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_STD)
                net_interfaces.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::TCP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_ULL)
                net_interfaces.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_ULL)
                net_interfaces.emplace_back(core);
        }
    }

    if (net_interfaces.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(net_interfaces[stack_index]);
}

void
configure(CmdLineIterator i, CmdLineIterator e,
          ni::net_interfaces & net_interfaces)
{
    if (i == e)
    {
        for (std::uint16_t index = 0, e = net_interfaces.count();
                index != e; ++index)
            net_interfaces.configure(index).v();
    }
    else
    {
        auto const index = std::atoi(*i);

        ensure_end(++i, e);

        net_interfaces.configure(index).v();
    }
}

void
unconfigure(CmdLineIterator i, CmdLineIterator e,
            ni::net_interfaces & net_interfaces)
{
    if (i == e)
    {
        for (std::uint16_t index = 0, e = net_interfaces.count();
                index != e; ++index)
            net_interfaces.unconfigure(index).v();
    }
    else
    {
        auto const index = std::atoi(*i);

        ensure_end(++i, e);

        net_interfaces.unconfigure(index).v();
    }
}

char const USAGE[] =
"Usage: " BIN_NAME " interface { ip | link }\n"
"       " BIN_NAME " interface [un]configure [ IF_INDEX ]\n";

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            ni::net_interfaces & net_interfaces)
{
    if (i == e)
    {
        throw std::runtime_error{"Missing SCOPE argument, "
                                 "try '" BIN_NAME " help'"};
    }
    else if (is_prefix(*i, "help"))
    {
        std::cout << USAGE << std::flush;
    }
    else if (is_prefix(*i, "ip"))
    {
        IpScope::run(++i, e, net_interfaces);
    }
    else if (is_prefix(*i, "link"))
    {
        LinkScope::run(++i, e, net_interfaces);
    }
    else if (is_prefix(*i, "unconfigure"))
    {
        unconfigure(++i, e, net_interfaces);
    }
    else if (is_prefix(*i, "configure"))
    {
        configure(++i, e, net_interfaces);
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown SCOPE '" << *i
              << "', try '" BIN_NAME " interface help'";
        throw std::runtime_error{error.str()};
    }
}

} // anonymous namespace

namespace InterfaceScope {

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    ni::net_interfaces net_interfaces{create_interfaces(root, stack_type, stack_index)};

    parse_scope(i, e, net_interfaces);
}

} // namespace InterfaceScope

ENYX_CORES_NAMESPACE_END
