#include "IcmpScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include <enyx/cores/hardware_ids.hpp>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace IcmpScope {

namespace {

namespace en = enyx;
namespace ni = en::icmp;

ni::icmp
create_icmp(hw::core & root,
           Application::StackType stack_type,
           std::uint32_t stack_index)
{
    std::vector<ni::icmp> icmp;

    for (auto & core : root)
    {
        auto const hardware_id = core.get_descriptor().hardware_id;
        if (hardware_id == enyx::hardware_ids::TCP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_STD)
                icmp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_STD)
                icmp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::TCP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_ULL)
                icmp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_ULL)
                icmp.emplace_back(core);
        }
    }

    if (icmp.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(icmp[stack_index]);
}

char const USAGE[] =
"Usage: " BIN_NAME " icmp show\n"
"       " BIN_NAME " icmp set server ( enable | disable )\n";

void
set_server_enable(CmdLineIterator i, CmdLineIterator e, ni::icmp & icmp)
{
    ensure_end(i, e);

    icmp.enable().value();
}

void
set_server_disable(CmdLineIterator i, CmdLineIterator e, ni::icmp & icmp)
{
    ensure_end(i, e);

    icmp.disable().value();
}

void
set_server(CmdLineIterator i, CmdLineIterator e, ni::icmp & icmp)
{
    if (i == e)
        throw std::runtime_error{"Expected { enable | disable }"};

    if (is_prefix(*i, "enable"))
        set_server_enable(++i, e, icmp);
    else if (is_prefix(*i, "disable"))
        set_server_disable(++i, e, icmp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
            << "', try '" BIN_NAME " icmp help'";
        throw std::runtime_error{error.str()};
    }
}

void
set(CmdLineIterator i, CmdLineIterator e, ni::icmp & icmp)
{
    if (i == e)
        throw std::runtime_error{"Expected server"};

    if (is_prefix(*i, "server"))
    {
        set_server(++i, e, icmp);
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " icmp help'";
        throw std::runtime_error{error.str()};
    }
}

void
show(CmdLineIterator i, CmdLineIterator e, ni::icmp & icmp)
{
    std::cout << "server: ";
    if (icmp.enabled())
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
    else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;
    std::cout << std::endl;
}

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            ni::icmp & icmp)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, icmp);
    else if (is_prefix(*i, "set"))
        set(++i, e, icmp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " icmp help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    ni::icmp icmp{create_icmp(root, stack_type, stack_index)};
    parse_scope(i, e, icmp);
}

} // namespace IcmpScope

ENYX_CORES_NAMESPACE_END
