#include "EthernetFlowControlScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include <enyx/cores/hardware_ids.hpp>
#include <enyx/cores/ethernet_flow_control/ethernet_flow_control.hpp>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace EthernetFlowControlScope {

namespace {

namespace en = enyx;
namespace ni = en::ethernet_flow_control;

ni::ethernet_flow_control
create_ethernet_flow_control(hw::core & root,
           Application::StackType stack_type,
           std::uint32_t stack_index)
{
    std::vector<ni::ethernet_flow_control> ethernet_flow_control;

    for (auto & core : root)
    {
        auto const hardware_id = core.get_descriptor().hardware_id;
        if (hardware_id == enyx::hardware_ids::UDP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_ULL)
                ethernet_flow_control.emplace_back(core);
        }
    }

    if (ethernet_flow_control.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(ethernet_flow_control[stack_index]);
}

char const USAGE[] =
"Usage: " BIN_NAME " ethernet_flow_control show\n"
"       " BIN_NAME " ethernet_flow_control ( enable | disable )\n"
"       " BIN_NAME " ethernet_flow_control set pause_threshold <THRESHOLD>\n"
"       " BIN_NAME " ethernet_flow_control set resume_threshold <THRESHOLD>\n"
"       " BIN_NAME " ethernet_flow_control set pause_time <VALUE>\n";

void
set_pause_time(CmdLineIterator i, CmdLineIterator e,
       ni::ethernet_flow_control & ethernet_flow_control)
{
    if (i == e)
        throw std::runtime_error{"Expected pause time value"};

    auto const value = std::atoi(*i);
    ensure_end(++i, e);

    ethernet_flow_control.set_pause_time(value).value();
}

void
set_resume_threshold(CmdLineIterator i, CmdLineIterator e,
       ni::ethernet_flow_control & ethernet_flow_control)
{
    if (i == e)
        throw std::runtime_error{"Expected resume threshold in bytes"};

    auto const threshold = std::atoi(*i);
    ensure_end(++i, e);

    ethernet_flow_control.set_resume_threshold(threshold).value();
}

void
set_pause_threshold(CmdLineIterator i, CmdLineIterator e,
       ni::ethernet_flow_control & ethernet_flow_control)
{
    if (i == e)
        throw std::runtime_error{"Expected pause threshold in bytes"};

    auto const threshold = std::atoi(*i);
    ensure_end(++i, e);

    ethernet_flow_control.set_pause_threshold(threshold).value();
}

void
set(CmdLineIterator i, CmdLineIterator e,
       ni::ethernet_flow_control & ethernet_flow_control)
{
    if (i == e)
        throw std::runtime_error{"Expected feature to set"};

    if (is_prefix(*i, "pause_threshold"))
    {
        set_pause_threshold(++i, e, ethernet_flow_control);
    }
    else if (is_prefix(*i, "resume_threshold"))
    {
        set_resume_threshold(++i, e, ethernet_flow_control);
    }
    else if (is_prefix(*i, "pause_time"))
    {
        set_pause_time(++i, e, ethernet_flow_control);
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " ethernet_flow_control help'";
        throw std::runtime_error{error.str()};
    }
}

void
enable(CmdLineIterator i, CmdLineIterator e,
       ni::ethernet_flow_control & ethernet_flow_control)
{
    ensure_end(i, e);

    ethernet_flow_control.enable().value();
}

void
disable(CmdLineIterator i, CmdLineIterator e,
        ni::ethernet_flow_control & ethernet_flow_control)
{
    ensure_end(i, e);

    ethernet_flow_control.disable().value();
}

void
show(CmdLineIterator i, CmdLineIterator e,
     ni::ethernet_flow_control & ethernet_flow_control)
{
    std::cout << "ethernet_flow_control: ";
    if (ethernet_flow_control.is_enabled()) {
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
        std::cout << "\nbuffer size: "
            << Color::BOLD
            << ethernet_flow_control.get_buffer_size().value()
            << Color::NONE << "B";
        std::cout << "\ninternal word size: "
            << Color::BOLD
            << ethernet_flow_control.get_internal_word_size().value()
            << Color::NONE << "B";
        std::cout << "\nused buffer size: "
            << Color::BOLD
            << ethernet_flow_control.get_used_buffer().value()
            << Color::NONE << "B";
        std::cout << "\ngenerated pause frame count: "
            << Color::BOLD
            << ethernet_flow_control.get_pause_frame_count().value()
            << Color::NONE;
        std::cout << "\npause threshold: "
            << Color::BOLD
            << ethernet_flow_control.get_pause_threshold().value()
            << Color::NONE << "B";
        std::cout << "\nresume threshold: "
            << Color::BOLD
            << ethernet_flow_control.get_resume_threshold().value()
            << Color::NONE << "B";
        std::cout << "\npause length: "
            << Color::BOLD << ethernet_flow_control.get_pause_time().value()
            << Color::NONE << " pause units (512 bit times)";
    } else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;
    std::cout << std::endl;
}

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            ni::ethernet_flow_control & ethernet_flow_control)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, ethernet_flow_control);
    else if (is_prefix(*i, "set"))
        set(++i, e, ethernet_flow_control);
    else if (is_prefix(*i, "enable"))
        enable(++i, e, ethernet_flow_control);
    else if (is_prefix(*i, "disable"))
        disable(++i, e, ethernet_flow_control);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '"
            BIN_NAME " ethernet_flow_control help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    ni::ethernet_flow_control ethernet_flow_control{
        create_ethernet_flow_control(root, stack_type, stack_index)};
    parse_scope(i, e, ethernet_flow_control);
}

} // namespace EthernetFlowControlScope

ENYX_CORES_NAMESPACE_END
