#include <stdbool.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>

#include <enyx/hw_c/core.h>

/**
 * Object representing the transceiver subsystem.
 */
typedef struct enyx_transceiver enyx_transceiver;

/**
 * Object representing a SFP transceiver.
 */
typedef struct enyx_sfp enyx_sfp;

/**
 * Object representing a QSFP transceiver.
 */
typedef struct enyx_qsfp enyx_qsfp;

/**
 * Supported SFP speed rates.
 */
typedef enum {
    ENYX_SFP_REDUCED_BANDWIDTH = 0,
    ENYX_SFP_FULL_BANDWIDTH = 1,
} enyx_sfp_rate;

/**
 * Create the transceiver subsystem from a root core.
 *
 * @param root The root core
 * @return The created transceiver object or NULL on error (with @b errno set
 * accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_transceiver *
enyx_transceiver_create(enyx_hw_core * root);

/**
 * Destroy a tranceiver.
 *
 * @param transceiver The transceiver to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_transceiver_destroy(enyx_transceiver * transceiver);

/**
 * Count the available SFP transceivers.
 *
 * @param transceiver The transceiver object
 * @return The number of SFP transceivers on success or -1 on error (with @b
 * errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_transceiver_count_sfp(enyx_transceiver const * transceiver);

/**
 * Count the available QSFP transceivers.
 *
 * @param transceiver The transceiver object
 * @return The number of QSFP transceivers on success or -1 on error (with @b
 * errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_transceiver_count_qsfp(enyx_transceiver const * transceiver);

/**
 * Get a SFP transciever handler object from a SFP port ID.
 *
 * @param transceiver The transceiver object
 * @param port The SFP port ID
 * @return The SFP object on success or NULL on error (with @b errno set
 * accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_sfp *
enyx_transceiver_get_sfp(enyx_transceiver * transceiver, uint8_t port);

/**
 * Get a QSFP transciever handler object from a QSFP port ID.
 *
 * @note This checks the ``ModPrsL`` QSFP pin to check if a QSFP module is
 *       plugged. If this pin is high (module not plugged), this will return
 *       an error with ``errno`` set to ``ENOSYS``.
 *
 * @param transceiver The transceiver object
 * @param port The QSFP port ID
 * @return The QSFP object on success or NULL on error (with @b errno set
 * accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_qsfp *
enyx_transceiver_get_qsfp(enyx_transceiver * transceiver, uint8_t port);

/**
 * Destroy the QSFP object.
 *
 * @param qsfp The QSFP object to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_qsfp_destroy(enyx_qsfp * qsfp);

/**
 * Get the QSFP port ID.
 *
 * @param qsfp The QSFP to use
 * @return The QSFP port ID on success, -1 on error (with @b errno set
 * accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_qsfp_get_id(enyx_qsfp const * qsfp);

/**
 * Check if low power mode is enabled.
 *
 * @note This checks the value of the ``LPMode`` QSFP pin and returns ``true``
 *       if the pin is high.
 *
 * @param qsfp The QSFP to use
 * @param enabled The power mode status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_is_low_power_mode_enabled(enyx_qsfp const * qsfp, bool * enabled);

/**
 * Enable low power mode.
 *
 * @note This sets the ``LPMode`` QSFP pin to low.
 *
 * @param qsfp The QSFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_enable_low_power_mode(enyx_qsfp * qsfp);

/**
 * Disable low power mode.
 *
 * @note This sets the ``LPMode`` QSFP pin to high.
 *
 * @param qsfp The QSFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_disable_low_power_mode(enyx_qsfp * qsfp);

/**
 * Check if I2C communication is enabled.
 *
 * @note This checks the value of the ``ModSelL`` QSFP pin and returns ``true``
 *       if the pin is low.
 *
 * @param qsfp The QSFP to use
 * @param enabled The I2C communication status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_is_i2c_communication_enabled(enyx_qsfp const * qsfp, bool * enabled);

/**
 * Enable I2C communication.
 *
 * @note This sets the ``ModSelL`` QSFP pin to low.
 *
 * @param qsfp The QSFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_enable_i2c_communication(enyx_qsfp * qsfp);

/**
 * Disable I2C communication.
 *
 * @note This sets the ``ModSelL`` QSFP pin to high.
 *
 * @param qsfp The QSFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_disable_i2c_communication(enyx_qsfp * qsfp);

/**
 * Reset the QSFP module.
 *
 * @note This sets the ``Reset`` QSFP pin to low, waits for the minimum pulse
 *       length (2us), then reverts the pin to high.
 *
 * @param qsfp The QSFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_reset_module(enyx_qsfp * qsfp);


/**
 * Check if a fault might be detected on the QSFP module.
 *
 * @note This checks the ``IntL`` QSFP pin and returs ``true`` (there is a
 *       possible module operational fault) when the pin is low.
 *
 * @param qsfp The QSFP to use
 * @param fault The fault status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_qsfp_possible_fault_detected(enyx_qsfp const * qsfp, bool * fault);

/**
 * Destroy the SFP object.
 *
 * @param sfp The SFP object to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_sfp_destroy(enyx_sfp * sfp);

/**
 * Get the SFP port ID.
 *
 * @param sfp The SFP to use
 * @return The SFP port ID on success, -1 on error (with @b errno set
 * accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_sfp_get_id(enyx_sfp const * sfp);

/**
 * Check if the SFP transmitter is enabled.
 *
 * @note This checks the ``TX Disable`` SFP pin and returns ``true`` when the
 *       pin is low (transmitter on).
 *
 * @param sfp The SFP to use
 * @param enabled The activation status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_is_transmitter_enabled(enyx_sfp const * sfp, bool * enabled);

/**
 * Enable the SFP transmitter.
 *
 * @note This sets the ``TX Disable`` SFP pin to low.
 *
 * @param sfp The SFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_enable_transmitter(enyx_sfp * sfp);

/**
 * Disable the SFP transmitter.
 *
 * @note This sets the ``TX Disable`` pin to high.
 *
 * @param sfp The SFP to use
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_disable_transmitter(enyx_sfp * sfp);

/**
 * Check the SFP for TX fault.
 *
 * @note This checks the ``TX Fault`` SFP pin and returns ``true`` when the pin
 *       is high (there is a laser fault of some kind).
 *
 * @param sfp The SFP to use
 * @param fault The SFP TX fault status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_get_tx_fault(enyx_sfp const * sfp, bool * fault);

/**
 * Get the SFP speed rate.
 *
 * @note This checks the ``Rate Select`` SFP pin and returns the corresponding
 *       bandwidth according to the specification.
 *
 * @param sfp The SFP to use
 * @param rate The SFP rate
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_get_rate(enyx_sfp const * sfp, enyx_sfp_rate * rate);

/**
 * Set the SFP speed rate.
 *
 * @note This sets the ``Rate Select`` SFP pin to select the chosen bandwidth.
 *
 * @param sfp The SFP to use
 * @param rate The SFP rate to set
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_set_rate(enyx_sfp * sfp, enyx_sfp_rate rate);

/**
 * Check the SFP RX signal.
 *
 * @note This checks the ``LOS`` (Loss of Signal) SFP pin and return ``true``
 *       when the pin is low (normal operation).
 *
 * @param sfp The SFP to use
 * @param signal The SFP RX signal status
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_sfp_has_signal(enyx_sfp const * sfp, bool * signal);
