#pragma once

#include <stddef.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/hw_c/core.h>

/**
 * Main test_point management object.
 */
typedef struct enyx_test_point enyx_test_point;

/**
 * @brief Get the number of transmitted packets.
 *
 * @param tp The test_point object to use.
 * @param count The packet count.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_packet_count(enyx_test_point const * tp, uint32_t * count);

/**
 * @brief Get the amount of transmitted bytes.
 *
 * @param tp The test_point object to use.
 * @param count The byte count.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_bytes_count(enyx_test_point const * tp, uint32_t * count);

/**
 * @brief Get the number of packets with errors.
 *
 * @param tp The test_point object to use.
 * @param count The error count.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_error_count(enyx_test_point const * tp, uint32_t * count);

/**
 * @brief Get the number of dropped packets.
 *
 * @param tp The test_point object to use.
 * @param count The drop count.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_drop_count(enyx_test_point const * tp, uint32_t * count);

/**
 * @brief Check if drop mode is enabled. If enabled, every packet going through
 * will be dropped.
 *
 * @param tp The test_point object to use.
 * @return 1 if drop mode is enabled, 0 if not, and -1 on error (with @b errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_is_drop_mode_enabled(enyx_test_point const * tp);

/**
 * @brief Enable drop mode. Every packet will be dropped.
 *
 * @param tp The test_point object to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_enable_drop_mode(enyx_test_point const * tp);

/**
 * @brief Disable drop mode. Packets will not be dropped.
 *
 * @param tp The test_point object to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_disable_drop_mode(enyx_test_point const * tp);

/**
 * @brief Check the link status.
 *
 * @param tp The test_point object to use.
 * @return 1 if link is up, 0 if not, and -1 on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_link_status(enyx_test_point const * tp);

/**
 * @brief Get the link loss count.
 *
 * @param tp The test_point object to use.
 * @param count The link loss count.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_get_link_loss_count(enyx_test_point const * tp, uint32_t * count);

/**
 * @brief Check if the drop generation is ready. Will be true if packet drop
 * generation is pending.
 *
 * @param tp The test_point object to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_is_ready_to_drop(enyx_test_point const * tp);

/**
 * @brief Check if the backpressure generation is ready. Will be true if
 * backpressure generation is pending.
 *
 * @param tp The test_point object to use.
 * @return 1 if ready, 0 if not, and -1 on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_is_ready_to_backpressure(enyx_test_point const * tp);

/**
 * @brief Check if the error generation is ready. Will be true if packet error
 * generation is pending.
 *
 * @param tp The test_point object to use.
 * @return 1 if ready, 0 if not, and -1 on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_is_ready_to_generate_errors(enyx_test_point const * tp)
    ENYX_CORES_C_DEPRECATED("replaced by enyx_test_point_is_ready_to_generate_errors");

/// @copydoc enyx_test_is_ready_to_generate_errors
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_is_ready_to_generate_errors(enyx_test_point const * tp);

/**
 * @brief Drop @b count packets.
 *
 * @param tp The test_point object to use.
 * @param count The number of packet to drop.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_generate_drops(enyx_test_point * tp, uint16_t count);

/**
 * @brief Generate backpressure for @b ticks.
 *
 * @param tp The test_point object to use.
 * @param ticks The duration of backpressure in ticks.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_generate_backpressure(enyx_test_point * tp, uint16_t ticks);

/**
 * @brief Put @b count packets in error.
 *
 * @param tp The test_point object to use.
 * @param count The number of packet to set in error.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_test_point_generate_errors(enyx_test_point * tp, uint16_t count);

/**
 * Create the test point subsystem from an avl_st_test_point core.
 *
 * @param core The core to use.
 * @return The subsystem on succes. NULL on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
enyx_test_point *
enyx_test_point_create(enyx_hw_core * core);

/**
 * Destroy a test point subsystem.
 *
 * @param tp The test point to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
void
enyx_test_point_destroy(enyx_test_point * tp);
