#pragma once

#include <cerrno>
#include <system_error>

#include <enyx/cores_c/udp/data_stream.h>

#include <enyx/cores/data_stream/sink.hpp>
#include <enyx/cores/data_stream/source.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_NAMESPACE_BEGIN
namespace udp {

/// C++ alias of @b C @ref enyx_udp_source_metadata
using source_metadata = enyx_udp_source_metadata;

/// C++ alias of @b C @ref enyx_udp_sink_metadata
using sink_metadata = enyx_udp_sink_metadata;

/**
 * C++ UDP source abstraction
 *
 * Usage:
 * @code
 * auto on_udp_data = [] (std::uint8_t const * data, std::uint32_t size,
 *                        void const * metadata) {
 *      std::cout << "UDP data received!" << std::endl;
 * };
 *
 * source udp(on_udp_data);
 * enyx_hw_source hw_source(stream, udp);
 *
 * do
 *     hw_source.poll_once().v();
 * while (! is_exit_requested);
 * @endcode
 */
class source
{
public:
    /**
     *  Constructs an UDP source calling @p handler on data.
     *
     *  @note Handler signature is:
     *      void(std::uint8_t const * data, std::uint32_t size,
     *           void const * metadata);
     */
    template<typename Handler>
    source(Handler && handler);

     /**
     * UDP source callback to invoke when data to be decoded is available.
     *
     * @param data: The UDP data to decode
     * @param size: Data size
     * @param metadata: Metadata associated with passed data
     */
    void
    operator()(std::uint8_t const * data,
               std::uint32_t size,
               void const * metadata);

    /**
     * Get the current source as a C source
     *
     * @return A C source instance
     */
    ::enyx_data_stream_source
    as_next_source() const noexcept;

private:
    data_stream::source_ptr next_source_;
};

/**
 *  @copybrief enyx_udp_create_sink
 *
 *  @param data_sink The sink of the UDP data
 *  @return the created sink
 */
result<data_stream::sink>
create_sink(data_stream::sink & data_sink);

} /* namespace udp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/udp/data_stream.ipp>
