#include <memory>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

#include <enyx/cores_c/transceiver/transceiver.h>

#include <enyx/hw/core.hpp>

#include <enyx/cores/macros.hpp>

ENYX_CORES_DEFAULT_DELETE(sfp)
ENYX_CORES_DEFAULT_DELETE(qsfp)

ENYX_CORES_NAMESPACE_BEGIN

namespace transceiver {

/**
 * @copydoc enyx_sfp
 */
class sfp {
public:
    /**
     * @copydoc enyx_sfp_rate
     */
    using sfp_rate = ::enyx_sfp_rate;

    sfp(std::shared_ptr<::enyx_transceiver> const & transceiver,
        ::enyx_sfp * sfp);

    /**
     * Get the SFP port ID.
     *
     * @return The SFP port ID on success or an error.
     */
    result<uint8_t>
    get_id() const;

    /**
     * Check if the SFP transmitter is enabled.
     *
     * @note This checks the ``TX Disable`` SFP pin and returns ``true`` when
     *       the pin is low (transmitter on).
     *
     * @return A result containing the activation status or an error.
     */
    result<bool>
    is_transmitter_enabled() const;

    /**
     * Enable the SFP transmitter.
     *
     * @note This sets the ``TX Disable`` SFP pin to low.
     *
     * @return A result containing an error on error
     */
    result<void>
    enable_transmitter();

    /**
     * Disable the SFP transmitter.
     *
     * @note This sets the ``TX Disable`` SFP pin to high.
     *
     * @return A result containing an error on error
     */
    result<void>
    disable_transmitter();

    /**
     * Check the SFP for TX fault.
     *
     * @note This checks the ``TX Fault`` SFP pin and returns ``true`` when the
     *       pin is high (there is a laser fault of some kind).
     *
     * @return A result containing the fault status or an error.
     */
    result<bool>
    get_tx_fault() const;

    /**
     * Get the SFP speed rate.
     *
     * @note This checks the ``Rate Select`` SFP pin and returns the
     *       corresponding bandwidth according to the specification.
     *
     * @return The SFP speed rate or an error.
     */
    result<sfp_rate>
    get_rate() const;

    /**
     * Set the SFP speed rate.
     *
     * @note This sets the ``Rate Select`` SFP pin to select the chosen
     *       bandwidth.
     *
     * @param rate The SFP rate to set
     * @return A result containing an error on error
     */
    result<void>
    set_rate(sfp_rate rate);

    /**
     * Check the SFP RX signal.
     *
     * @note This checks the ``LOS`` (Loss of Signal) SFP pin and return
     *       ``true`` when the pin is low (normal operation).
     *
     * @return The SFP RX signal status or an error.
     */
    result<bool>
    has_signal() const;

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_sfp *
    handle();

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_sfp const *
    handle() const;

private:
    std::shared_ptr<::enyx_transceiver> transceiver_;
    std::unique_ptr<::enyx_sfp> sfp_;
};

/**
 * @copydoc enyx_qsfp
 */
class qsfp {
public:
    /**
     * Construct a QSFP object from a C transceiver and a C QSFP.
     *
     * @note This whould not be used directly, and is not part of the stable
     * API. transceiver::get_qsfp() should be used instead.
     *
     * @param transceiver A shared pointer to an already contructed transceiver
     * @param qsfp A pointer to a constructed C QSFP to take ownership from
     */
    qsfp(std::shared_ptr<::enyx_transceiver> const & transceiver,
         ::enyx_qsfp * qsfp);

    /**
     * Get the QSFP port ID.
     *
     * @return The QSFP port ID on success or an error.
     */
    result<uint8_t>
    get_id() const;

    /**
     * Check if low power mode is enabled.
     *
     * @note This checks the value of the ``LPMode`` QSFP pin and returns
     *       ``true`` if the pin is high.
     *
     * @return A result containing the low power mode status on success, or an
     * error
     */
    result<bool>
    is_low_power_mode_enabled();

    /**
     * Enable low power mode.
     *
     * @note This sets the ``LPMode`` QSFP pin to low.
     *
     * @return A result containing an error on error.
     */
    result<void>
    enable_low_power_mode();

    /**
     * Disable low power mode.
     *
     * @note This sets the ``LPMode`` QSFP pin to high.
     *
     * @return A result containing an error on error.
     */
    result<void>
    disable_low_power_mode();

    /**
     * Check if I2C communication is enabled.
     *
     * @note This checks the value of the ``ModSelL`` QSFP pin and returns
     *       ``true`` if the pin is low.
     *
     * @return A result containing the I2C communication status on success, or
     * an error
     */
    result<bool>
    is_i2c_communication_enabled();

    /**
     * Enable I2C communication.
     *
     * @note This sets the ``ModSelL`` QSFP pin to low.
     *
     * @return A result containing an error on error.
     */
    result<void>
    enable_i2c_communication();

    /**
     * Disable I2C communication.
     *
     * @note This sets the ``ModSelL`` QSFP pin to high.
     *
     * @return A result containing an error on error.
     */
    result<void>
    disable_i2c_communication();

    /**
     * Reset the QSFP module.
     *
     * @note This sets the ``Reset`` QSFP pin to low, waits for the minimum
     *       pulse length (2us), then reverts the pin to high.
     *
     * @return A result containing an error on error.
     */
    result<void>
    reset_module();

    /**
     * Check if a fault might be detected on the QSFP module.
     *
     * @note This checks the ``IntL`` QSFP pin and returs ``true`` (there is a
     *       possible module operational fault) when the pin is low.
     *
     * @return A result containing the fault status on succes, or an error.
     */
    result<bool>
    possible_fault_detected() const;

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_qsfp *
    handle();

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_qsfp const *
    handle() const;

private:
    std::shared_ptr<::enyx_transceiver> transceiver_;
    std::unique_ptr<::enyx_qsfp> qsfp_;
};

/**
 * @copydoc enyx_transceiver
 */
class transceiver {
public:
    /**
     * Construct the transceiver subsystem from a root core.
     *
     * @param root The root core
     */
    transceiver(hw::core & root);

    /**
     * Count the available SFP transceivers.
     *
     * @return A result containing the number of SFP transceiver on success or
     * an error.
     */
    result<size_t>
    count_sfp() const;

    /**
     * Count the available QSFP transceivers.
     *
     * @return A result containing the number of QSFP transceiver on success or
     * an error.
     */
    result<size_t>
    count_qsfp() const;

    /**
     * Get a SFP transceiver object from a SFP port ID.
     *
     * @param port The SFP port ID
     * @return A result containing the SFP object on success or an error
     */
    result<sfp>
    get_sfp(uint8_t port);

    /**
     * Get a QSFP transceiver object from a QSFP port ID.
     *
     * @note This checks the ``ModPrsL`` QSFP pin to check if a QSFP module is
     *       plugged. If this pin is high (module not plugged), this will return
     *       an error with ``errno`` set to ``ENOSYS``.
     *
     * @param port The QSFP port ID
     * @return A result containing the QSFP object on success or an error
     */
    result<qsfp>
    get_qsfp(uint8_t port);

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_transceiver *
    handle();

    /**
     * Get the C handle.
     *
     * @return The C handle
     */
    ::enyx_transceiver const *
    handle() const;

private:
    std::shared_ptr<::enyx_transceiver> transceiver_;
};

} /* namespace transceiver */

ENYX_CORES_NAMESPACE_END

#include "transceiver.ipp"
