#pragma once

#include <enyx/cores_c/tcp/context.h>

#include <memory>
#include <system_error>
#include <utility>

#include <enyx/cores/namespace.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_tcp_context>
{
    void
    operator()(::enyx_tcp_context * ptr) const
    {
        ::enyx_tcp_context_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

/// @copydoc enyx_tcp_event_header
using event_header = ::enyx_tcp_event_header;

/// @copydoc enyx_tcp_event_type
using event_type = ::enyx_tcp_event_type;

/// @copydoc enyx_tcp_credit_event
using credit_event = ::enyx_tcp_credit_event;

/// @copydoc enyx_tcp_session_status_event
using session_status_event = ::enyx_tcp_session_status_event;

/// @copydoc enyx_tcp_state
using state = ::enyx_tcp_state;

/// @copydoc enyx_tcp_state_info
using state_info = ::enyx_tcp_state_info;

/// @cond
class abstract_wrapper
{
public:
    static void
    on_state_change(std::uint32_t session_id,
                    state new_state,
                    state_info const * state_info,
                    void * opaque)
    {
        auto thiz = reinterpret_cast<abstract_wrapper *>(opaque);
        thiz->call_on_state_change(session_id, new_state, state_info);
    }

    static void
    on_credit_change(std::uint32_t session_id,
                     std::uint32_t credit,
                     void * opaque)
    {
        auto thiz = reinterpret_cast<abstract_wrapper *>(opaque);
        thiz->call_on_credit_change(session_id, credit);
    }

    static void
    on_emi_error(void * opaque)
    {
        auto thiz = reinterpret_cast<abstract_wrapper *>(opaque);
        thiz->call_on_emi_error();
    }

private:
    virtual void
    call_on_state_change(std::uint32_t session_id,
                         state new_state,
                         state_info const * state_info) = 0;

    virtual void
    call_on_credit_change(std::uint32_t session_id,
                          std::uint32_t credit) = 0;

    virtual void
    call_on_emi_error() = 0;
};

template<typename OnStateChange, typename OnCreditChange, typename OnEmiError>
class wrapper : public abstract_wrapper
{
public:
    wrapper(OnStateChange && on_state_change,
            OnCreditChange && on_credit_change,
            OnEmiError && on_emi_error)
        : on_state_change_(std::forward<OnStateChange>(on_state_change))
        , on_credit_change_(std::forward<OnCreditChange>(on_credit_change))
        , on_emi_error_(std::forward<OnEmiError>(on_emi_error))
    { }

private:
    virtual void
    call_on_state_change(std::uint32_t session_id,
                    state new_state,
                    state_info const * state_info) override
    {
        on_state_change_(session_id, new_state, state_info);
    }

    virtual void
    call_on_credit_change(std::uint32_t session_id,
                     std::uint32_t credit) override
    {
        on_credit_change_(session_id, credit);
    }

    virtual void
    call_on_emi_error() override
    {
        on_emi_error_();
    }

    OnStateChange on_state_change_;
    OnCreditChange on_credit_change_;
    OnEmiError on_emi_error_;
};
/// @endcond

/**
 * C++ TCP context abstraction
 *
 * Usage:
 * @code
 * auto on_state_change = [] (std::uint32_t session_id,
 *                            enyx::tcp::context::state new_state,
 *                            enyx::tcp::context::state_info const* state_info) {
 *      std::cout << "state changed" << std::endl;
 * };
 *
 * auto on_credit_change = [] (std::uint32_t session_id,
 *                             std::uint32_t credit) {
 *      std::cout << "credit updated" << std::endl;
 * };
 *
 * auto on_error = [] {
 *      std::cout << "error occured" << std::endl;
 * };
 *
 * context tcp_context{on_state_change, on_credit_change, on_error};
 * enyx_hw_source hw_source{stream, tcp_context};
 *
 * do
 *     hw_source.poll_once().v();
 * while (! is_exit_requested);
 * @endcode
 */
class context final
{
public:
    /// @copydoc enyx_tcp_state
    using state = ::enyx_tcp_state;

    /// @copydoc enyx_tcp_state_info
    using state_info = ::enyx_tcp_state_info;

public:
    /**
     *  Constructs a TCP EMI stream context calling @p on_state_change,
     *  @p on_credit_change on decoded data, and @p on_emi_error on invalid data.
     *
     *  @copybrief enyx_tcp_context_create
     *
     *  @tparam OnStateChange Type of the on_state_change callback
     *  @param on_state_change The callback to invoke on state change
     *  @tparam OnCreditChange Type of the on_credit_change callback
     *  @param on_credit_change The callback to invoke on credit change
     *  @tparam OnEmiError Type of the on_emi_error callback
     *  @param on_emi_error The callback to invoke on EMI error
     *  @return 0 on success, -1 on error (@b errno is set accordingly)
     *
     *  @note OnStateChange signature is:
     *      void(std::uint32_t session_id,
     *           enyx::tcp::context::state new_state,
     *           enyx::tcp::context::state_info const* state_info);
     *
     *  @note OnCreditChange signature is:
     *      void(std::uint32_t session_id,
     *           std::uint32_t credit);
     *
     *  @note OnEmiError signature is:
     *      void();
     */
    template<typename OnStateChange, typename OnCreditChange, typename OnEmiError>
    context(OnStateChange && on_state_change,
            OnCreditChange && on_credit_change,
            OnEmiError && on_emi_error);

    /**
     * TCP context callback to invoke when EMI data is available.
     *
     * @param data: The EMI data to decode
     * @param size: Data size
     * @param metadata: Metadata associated with passed data
     */
    void
    operator()(std::uint8_t const * data,
               std::uint32_t size,
               void const * metadata);

    /**
     * Get the current context as a C source
     *
     * @return A C source instance
     */
    ::enyx_data_stream_source
    as_next_source() const noexcept;

    /**
     * Retrieve the @b C handle associated with this context.
     *
     * @return The C handle
     */
    ::enyx_tcp_context *
    handle() const noexcept;

private:
    std::unique_ptr<abstract_wrapper> wrapper_;
    std::unique_ptr<::enyx_tcp_context> io_context_;
};

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/tcp/context.ipp>
