/**
 *  @file
 *
 *  Contains the types and functions related to the Probe.
 */
#pragma once

#include <enyx/cores_c/probes/probe.h>

#include <memory>
#include <utility>
#include <iostream>
#include <cerrno>
#include <system_error>

#include <enyx/hw/result.hpp>
#include <enyx/hw/core.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

#include <enyx/cores/probes/event.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_probes_probe>
{
    void
    operator()(::enyx_probes_probe * ptr) const
    {
        ::enyx_probes_probe_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/**
 * @copydoc enyx_probes_probe
 *
 * @since 6.7.0
 */
class probe final
{
public:
    /// @copydoc enyx_probes_ts_format
    using id = ::enyx_probes_probe_id;

/// @cond
public:
    friend std::ostream &
    operator<<(std::ostream & out, probe const& p);
/// @endcond

public:
    /**
     * Construct a probe.
     *
     * @param core The probe core
     * @param new_id The probe ID to assign
     * @param ts_format The probe timestamp format
     */
    probe(hw::core const & core, id new_id, ::enyx_probes_ts_format ts_format);

    /**
     * Construct a Probe from its C handle.
     *
     * @note This will take the ownership of the C probe object, and it will
     * be destroyed upton this C++ object deletion.
     *
     * @param root The root core.
     * @param probe The probe C handle.
     *
     * @since 6.7.0
     */
    probe(hw::core const & root, ::enyx_probes_probe * probe);

    /**
     * Direct access to the underlying C enyx_probes_probe object.
     *
     * @return The C enyx_probes_probe object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_probe *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_probes_probe object.
     *
     * @return The C enyx_probes_probe object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_probe const *
    handle() const noexcept;

    /**
     * Retrieve a probe id.
     *
     * @return the id as integer.
     *
     * @since 6.7.0
     */
    result<id>
    get_id() const noexcept;

    /**
     * Retrieve the reference id field size.
     *
     * @return the size in byte.
     *
     * @since 6.7.0
     */
    event_format
    get_event_format() const noexcept;

    /**
     * Retrieve the probe name.
     *
     * @return the name string.
     *
     * @since 6.7.0
     */
    std::string
    get_name() const noexcept;

    /**
     * Retrieve the associated collector name.
     *
     * @return the name string.
     *
     * @since 6.7.0
     */
    std::string
    get_collector_name() const noexcept;

    /**
     * Retrieve the probe status.
     *
     * @return true if enabled, false if disabled.
     *
     * @since 6.7.0
     */
    bool
    is_enabled() const noexcept;

    /**
     * Enable the probe.
     *
     * @return A result to check if the call is successful.
     *
     * @since 6.7.0
     */
    result<void>
    enable() noexcept;

    /**
     * Disable the probe.
     *
     * @return A result to check if the call is successful.
     *
     * @since 6.7.0
     */
    result<void>
    disable() noexcept;

    /**
     * Retrieve the probe dropped event(s) count.
     *
     * @return A result containing the dropped event(s) count on success, or an
     * error on failure.
     *
     * @since 6.7.0
     */
    result<std::size_t>
    get_drop_count() noexcept;

    /**
     * Retrieve the probe event packets(s) count.
     *
     * @return A result containing the event(s) count on success, or an error
     * on failure.
     *
     * @since 6.7.0
     */
    result<std::size_t>
    get_packet_count() noexcept;

private:
    hw::core core_;
    std::unique_ptr<::enyx_probes_probe> probe_;
};

/**
 * Print the probe @p m into the output stream @p out
 *
 * @param out The output stream
 * @param p The probe to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, probe const& p);

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/probe.ipp>
