#pragma once

#include <memory>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/hw/core.hpp>

#include <enyx/cores_c/mm_from_st/mm_from_st.h>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_mm_from_st>
{
    void
    operator()(::enyx_mm_from_st * ptr) const
    {
        ::enyx_mm_from_st_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace mm_from_st {

/**
 * @copydoc ::enyx_mm_from_st
 */
class mm_from_st {
public:
    /**
     * Create a MM from ST object from an avl_mm_from_st core.
     *
     * @param core The core to use
     *
     * @throw std::system_error on failure.
     */
    mm_from_st(enyx::hw::core & core);

    /**
     * Read a data packet.
     *
     * @param buffer The buffer to fill
     * @param size The buffer size
     * @return A result containing the amount of read data in case of success, or
     *         an error.
     * @note The returned packet size can be bigger than the given buffer size. In
     *       that case, the remaining data is flushed.
     */
    result<size_t>
    read(uint8_t * buffer, size_t size) noexcept;

    /**
     * Get the MTU.
     *
     * @return A result containing the MTU on success, or an error
     */
    result<size_t>
    get_mtu() const noexcept;

    /**
     * Get the data width.
     *
     * @return A result containing the data width on success, or an error
     */
    result<size_t>
    get_data_width() const noexcept;

    /**
     * Get the ID width.
     *
     * @return A result containing the ID width on success, or an error
     */
    result<size_t>
    get_id_width() const noexcept;

    /**
     * Get a const pointer to the underlying @b C handle.
     *
     * @return A const pointer to a @ref enyx_mm_from_st
     */
    enyx_mm_from_st const *
    handle() const noexcept;

    /**
     * Get a pointer to the underlying @b C handle.
     *
     * @return A pointer to a @ref enyx_mm_from_st
     */
    enyx_mm_from_st *
    handle() noexcept;

private:
    std::unique_ptr<enyx_mm_from_st> mm_from_st_c_;
};

} /* namespace mm_from_st */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/mm_from_st/mm_from_st.ipp>
