#pragma once

#include <enyx/cores_c/igmp/igmp.h>

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_igmp>
{
    void
    operator()(::enyx_igmp * ptr) const
    {
        ::enyx_igmp_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace igmp {

/**
 * @copybrief enyx_igmp
 *
 * This wrapper provides @b RAII of @b C @ref enyx_igmp
 */
class igmp
{
public:
    /**
     * @brief Construct the IGMP subsystem.
     *
     * This subsystem configures the following hw::core:
     *
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     *
     * @param core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    igmp(enyx::hw::core const& core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * @copybrief enyx_igmp_enabled
     *
     * @return true if IGMP is enabled.
     */
    bool
    enabled() const noexcept;

    /**
     * @copybrief enyx_igmp_enable
     *
     * @return A result object containg an error on failure
     */
    result<void>
    enable() noexcept;

    /**
     * @copybrief enyx_igmp_disable
     *
     * @return A result object containg an error on failure
     */
    result<void>
    disable() noexcept;

    /**
     * Access to the C handle.
     *
     * @return a const pointer to the C handle
     */
    ::enyx_igmp const *
    handle() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return a pointer to the C handle
     */
    ::enyx_igmp *
    handle() noexcept;

private:
    enyx::hw::core igmp_core_;
    std::unique_ptr<::enyx_igmp> handle_;
};

} /* namespace igmp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/igmp/igmp.ipp>
