#pragma once

#include <cstdint>
#include <cstring>

#include <memory>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores_c/hw_top/hw_top.h>

ENYX_CORES_NAMESPACE_BEGIN

namespace hw_top {

class serial_id final
{
public:
    static constexpr size_t len = 12;

    /**
     * Construct an uninitialized serial ID.
     */
    serial_id() noexcept;

    /**
     * Construct a serial ID from a @b C serial id object
     *
     * @param serial_id The @b C serial id object to construct from
     */
    serial_id(::enyx_hw_top_fpga_serial_id const& serial_id) noexcept;

    /**
     * Convert the serial id object into an Array
     *
     * @note This is useful for comparing serial ids.
     *
     * @return An array containing the raw serial id.
     */
    std::array<std::uint8_t, len>
    array() const noexcept;

    /**
     * Convert a @b C++ serial id object to a @b C serial id object
     *
     * @return The @b C serial id object
     */
    operator ::enyx_hw_top_fpga_serial_id * () noexcept;

    /**
     * Convert a @b C++ serial id object to a @b C serial id object
     *
     * @return The @b C serial id object
     */
    operator ::enyx_hw_top_fpga_serial_id const* () const noexcept;

private:
    ::enyx_hw_top_fpga_serial_id id_;
};

/**
 * Get a string representation of this serial ID as stream.
 *
 * @return The serial ID as stream.
 */
template<int len>
inline std::ostream&
operator<<(std::ostream& out, serial_id const& serial);

/**
 * Compare serial IDs @p a & @p b for equality
 *
 * @param a The first serial ID
 * @param b The second serial ID
 * @return true if both @p a & @p b are equal, false otherwise
 */
template<int len>
bool
operator==(serial_id const& a, serial_id const& b) noexcept;

/**
 * Compare serial IDs @p a & @p b for difference
 *
 * @param a The first serial ID
 * @param b The second serial ID
 * @return true if both @p a & @p b are different, false otherwise
 */
template<int len>
bool
operator!=(serial_id const& a, serial_id const& b) noexcept;

} /* namespace hw_top */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/hw_top/serial_id.ipp>
