#pragma once

#include <stdint.h>
#include <unistd.h>
#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/register_description.h>

/**
 * The address space of a core
 */
typedef struct {
    /// The memory zone corresponding to the core address space
    uint8_t * data;
    /// The size of data
    size_t size;
} enyx_hwm_core_address_space;

/**
 * A mocked register
 */
typedef struct {
    /**
     * A user defined object to be passed to `on_read` & `on_write` callbacks.
     */
    void * context;

    /**
     * Called on every read of the register.
     *
     * Default callaback calls `enyx_hwm_register_get()`.
     *
     * @param context The context of the register
     * @param[out] value The value to be read
     * @return 0 on success or -1 on failure (with @b errno set accordingly).
     */
    int
    (*on_read)(void * context, uint64_t * value);

    /**
     * Called on every write of the register.
     *
     * Default callaback calls `enyx_hwm_register_set()`.
     *
     * @param context The context of the register
     * @param value The value to be written
     * @return 0 on success or -1 on failure (with @b errno set accordingly).
     */
    int
    (*on_write)(void * context, uint64_t value);

    /// The register description
    enyx_hw_register_description description;

    /// The parent core address space
    enyx_hwm_core_address_space * core_address_space;
} enyx_hwm_register;

/**
 * Set the value of a register, without triggering the write callback
 *
 * @param reg The register to set
 * @param value The value to set
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_register_set(enyx_hwm_register * reg, uint64_t value);

/**
 * Get the value of a register, without triggering the read callback
 *
 * @param reg The register to get
 * @param[out] value The value to get
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_register_get(enyx_hwm_register const * reg,
                      uint64_t * value);

/**
 * Trigger the write callbacks of a register
 *
 * @param reg The register to trigger
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_register_trigger_write(enyx_hwm_register * reg);

/**
 * Trigger the read callbacks of a register
 *
 * @param reg The register to trigger
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_register_trigger_read(enyx_hwm_register * reg);

/**
 * Update the value register of a string register combo when the index register
 * changes.
 *
 * @note This function is a helper for bindings and generated code, it should
 * not be used in any other situation
 *
 * @param string The string to write
 * @param string_size the size of @p string
 * @param chunk_index The currend index of the string read
 * @param chunk_size The size in bytes of each read
 * @param value_reg The value register that will contain the next chunk
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_register_update_str(char const * string, size_t string_size,
                             size_t chunk_index, size_t chunk_size,
                             enyx_hwm_register * value_reg);
