/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  cores registers.
 */
#pragma once

#include <enyx/hw_c/register_io.h>

#include <string>
#include <chrono>
#include <system_error>

#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/register_description.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * Write the @p value into register @p r of core @p c
 * and store any error into @p failure.
 *
 * @param c The core to target.
 * @param r The register to write to.
 * @param value The integral value write (up to 64 bits).
 * @returns A result object containing an error if any occured.
 *
 * @since 5.0.0
 */
result<void>
write(core const& c, register_description const& r, std::uint64_t value) noexcept;

/**
 * Read the @p value from register @p r of core @p c
 * and store any error into @p failure.
 *
 * @param c The core to target.
 * @param r The register to read from.
 * @returns A result object containing either the read value or an error if any.
 *
 * @since 5.0.0
 */
result<std::uint64_t>
read(core const& c, register_description const& r) noexcept;

/**
 * Read the string exposed by register couple @p index & @p value
 * located within core @p c.
 *
 * @param c The core to target.
 * @param index The register used to select.
 * @param value The integral used to read value.
 * @returns A result object containing either the read string or an error if any.
 *
 * @since 5.0.0
 */
result<std::string>
read_string(core const& c,
            register_description const& index,
            register_description const& value) noexcept;

/**
 * Poll a register @p r until the read value is equal to @p expected.
 *
 * @param c The core to target
 * @param r The register to poll
 * @param expected The expected value
 * @param timeout_ms The millisecond timeout
 * @returns A result object containing an error if any occured.
 *
 * @since 5.0.0
 */
result<void>
poll_until_equal_to(core const& c,
                    register_description const& r,
                    std::uint64_t expected,
                    std::chrono::milliseconds timeout_ms) noexcept;

/**
 * Change a hardware page inside a core and wait until the page has successfuly
 * been changed.
 *
 * @param core The core containining the page.
 * @param page_count The page_cnt register of the page.
 * @param page_index The page_id register of the page.
 * @param page_ready The page_rdy register of the page (polled to check if a
 *                   page has succesfuly been changed).
 * @param new_page_id The new page id to set.
 * @return a result object containing an error if any occured.
 * @note This function blocks until the page has changed.
 *
 * @since 5.0.0
 */
result<void>
change_page(core const & core,
            register_description const & page_count,
            register_description const & page_index,
            register_description const & page_ready,
            std::uint64_t new_page_id);

/**
 * Change a hardware page inside a core and do not wait for page to be ready.
 *
 * @param core The core containining the page.
 * @param page_count The page_cnt register of the page.
 * @param page_index The page_id register of the page.
 * @param new_page_id The new page id to set.
 * @return a result object containing an error if any occured.
 * @note This function blocks until the page has changed.
 *
 * @since 5.1.0
 */
result<void>
change_page(core const & core,
            register_description const & page_count,
            register_description const & page_index,
            std::uint64_t new_page_id);

ENYX_HW_NAMESPACE_END

#include <enyx/hw/register_io.ipp>
