#pragma once

#include <cstdint>

#include <memory>
#include <string>
#include <system_error>
#include <vector>

#include <enyx/hw/macros.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/hw_c/properties.h>
#include <enyx/hw_c/filter.h>

ENYX_HW_NAMESPACE_BEGIN

/**
 * Opaque class representing properties.
 *
 * This class can be used as a filter to find specific device(s)
 * or as a source of information on a discovered device(s).
 *
 * @since 5.0.0
 */
class properties
{
public:
    /**
     * Create a properties from C const properties.
     *
     * @since 5.0.0
     */
    properties(enyx_hw_properties const *props);

    /// @name Index methods
    /// @{
    /**
     * Get the @p Property value.
     *
     * If the property is not available on this instance,
     * the @p failure parameter will be set accordingly.
     *
     * Otherwise, the @p property parameter will be filled with
     * the current value.
     *
     * @tparam Property The property type to get
     * @return A result object containing either the value of the @p Property
     *         or an error.
     *
     * @since 5.0.0
     */
    template<typename Property>
    result<typename Property::value_type>
    get() const noexcept;

    /**
     * Direct access to the underlying C properties object.
     *
     * @return The C properties object.
     *
     * @since 5.0.0
     */
    enyx_hw_properties const *
    handle() const noexcept;

    /// @}
    //
private:
    enyx_hw_properties const * properties_;
};

/**
 * Opaque class representing filter.
 *
 * This class can be used as a filter to find specific device(s)
 * or as a source of information on a discovered device(s).
 *
 * @since 5.0.0
 */
class filter
{
public:
    /**
     * Create a filter from C filter.
     *
     * @since 5.0.0
     */
    filter(enyx_hw_filter *f);

    /**
     * Create an empty filter instance
     *
     * @since 5.0.0
     */
    filter();

    /**
     * Create a filter instance with @p properties
     *
     * @tparam Properties The filter types
     * @param props The properties values
     *
     * @since 5.0.0
     */
    template<typename ...Properties>
    explicit
    filter(Properties... props);

    /// @name Index methods
    /// @{
    /**
     * Get the @p Property value.
     *
     * If the property is not available on this instance,
     * the @p failure parameter will be set accordingly.
     *
     * Otherwise, the @p property parameter will be filled with
     * the current value.
     *
     * @tparam Property The property type to get
     * @return A result object containing either the value of the @p Property
     *         or an error.
     *
     * @since 5.0.0
     */
    template<typename Property>
    result<typename Property::value_type>
    get() const noexcept;

    /**
     * Set the @p Property value
     *
     * If the property is not available on this instance the result will
     * contain an error.
     *
     * @tparam Property The property type to get
     * @param value The value of the property to set
     * @return A result object that may contain an error and that can be used
     *         to throw on failure.
     *
     * @since 5.0.0
     */
    template<typename Property>
    result<void>
    set(typename Property::value_type const& value) noexcept;

    /**
     * Set the @p Property value
     *
     * If the property is not available on this instance the result will
     * contain an error.
     *
     * @tparam Property The property type to get.
     * @param property The property object to set.
     * @return A result object that may contain an error and that can be used
     *         to throw on failure.
     *
     * @since 5.0.0
     */
    template<typename Property>
    result<void>
    set(Property const& property) noexcept;

    /**
     * Direct access to the underlying C filter object.
     *
     * @return The C filter object.
     *
     * @since 5.0.0
     */
    enyx_hw_filter *
    handle() noexcept;

    /**
     * Direct access to the underlying C filter object.
     *
     * @return The C filter object.
     *
     * @since 5.0.0
     */
    enyx_hw_filter const *
    handle() const noexcept;

    /// @}
    //
private:
    template<typename Property, typename... Properties>
    result<void>
    set(Property const& property, Properties... properties) noexcept;

private:
    std::unique_ptr<enyx_hw_filter, decltype(&enyx_hw_filter_destroy)> filter_;
};

/**
 *  Filter resources using their uid
 *
 *  @since 5.1.0
 */
struct uid
{
    /**
     * @brief The type of uid values.
     *
     * @since 5.1.0
     */
    using value_type = std::string;

    /**
     * The maximum size for the uid property.
     *
     * @since 5.1.0
     */
    static constexpr auto MAX_NAME_SIZE = 1024;

    /**
     * Construct an uid property.
     *
     * @param v The value of the uid property.
     * @since 5.1.0
     */
    uid(value_type const & v) : value(v) {};

    /**
     * Get the value of the uid property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.1.0
     */
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        std::vector<char> ret(uid::MAX_NAME_SIZE, 0);
        if (enyx_hw_prop_get_uid(props, ret.data(), ret.size()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return ret.data();
    }

    /**
     * Get the value of the uid property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.1.0
     */
    static result<value_type>
    get(enyx_hw_filter *filter)
    {
        std::vector<char> ret(uid::MAX_NAME_SIZE, 0);
        if (enyx_hw_filter_get_uid(filter, ret.data(), ret.size()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return ret.data();
    }

    /**
     * Set the value of the uid property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.1.0
     */
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
        if (enyx_hw_filter_set_uid(filter, value.c_str()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.1.0
     */
    const value_type value;
};

/**
 * Filter resources using their index
 *
 * @since 5.0.0
 */
struct index
{
    /**
     * @brief The type of index values.
     *
     * @since 5.0.0
     */
    using value_type = std::uint32_t;

    /**
     * Construct an index property.
     *
     * @param v The value of the index property.
     * @since 5.0.0
     */
    explicit
    index(value_type const & v) : value(v) {};

    /**
     * Get the value of the index property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        value_type ret;
        if (enyx_hw_prop_get_index(props, &ret) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * Get the value of the index property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    static
    result<value_type> get(enyx_hw_filter *filter)
    {
        value_type ret;
        if (enyx_hw_filter_get_index(filter, &ret) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * Set the value of the index property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.0.0
     */
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
        if (enyx_hw_filter_set_index(filter, value) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.0.0
     */
    value_type value;
};

/**
 *  Filter resources using their name
 *
 *  @since 5.0.0
 */
struct name
{
    /**
     * @brief The type of name values.
     *
     * @since 5.0.0
     */
    using value_type = std::string;

    /**
     * The maximum size for the name property.
     *
     * @since 5.0.0
     */
    static constexpr auto MAX_NAME_SIZE = 1024;

    /**
     * Construct an name property.
     *
     * @param v The value of the name property.
     * @since 5.0.0
     */
    name(value_type const & v) : value(v) {};

    /**
     * Get the value of the name property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        std::vector<char> ret(name::MAX_NAME_SIZE, 0);
        if (enyx_hw_prop_get_name(props, ret.data(), ret.size()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return ret.data();
    }

    /**
     * Get the value of the name property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    static result<value_type>
    get(enyx_hw_filter *filter)
    {
        std::vector<char> ret(name::MAX_NAME_SIZE, 0);
        if (enyx_hw_filter_get_name(filter, ret.data(), ret.size()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return ret.data();
    }

    /**
     * Set the value of the name property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.0.0
     */
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
        if (enyx_hw_filter_set_name(filter, value.c_str()) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.0.0
     */
    const value_type value;
};

/**
 *  Filter resources using their mtu
 *
 *  @since 5.0.0
 */
struct mtu
{
    /**
     * @brief The type of mtu values.
     *
     * @since 5.0.0
     */
    using value_type = std::uint32_t;

    /**
     * @deprecated replaced by a2c/c2a stream get_mtu methods
     * construct an mtu property.
     *
     * @param v the value of the mtu property.
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_mtu method")
    mtu(value_type const & v) : value(v) {};

    /**
     * @deprecated replaced by a2c/c2a stream get_mtu methods
     *
     * Get the value of the mtu property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_mtu method")
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        value_type ret;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_prop_get_mtu(props, &ret) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * @deprecated replaced by a2c/c2a stream get_mtu methods
     *
     * Get the value of the mtu property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_mtu method")
    static result<value_type>
    get(enyx_hw_filter *filter)
    {
        value_type ret;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_filter_get_mtu(filter, &ret) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * @deprecated replaced by a2c/c2a stream get_mtu methods
     *
     * Set the value of the mtu property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_mtu method")
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_filter_set_mtu(filter, value) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.0.0
     */
    const value_type value;
};


/**
 *  Filter resources using their size
 *
 *  @since 5.0.0
 */
struct size
{
    /**
     * @brief The type of size values.
     *
     * @since 5.0.0
     */
    using value_type = std::size_t;

    /**
     * @deprecated replaced by a2c/c2a stream get_size methods
     * construct an size property.
     *
     * @param v the value of the size property.
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_size method")
    size(value_type const & v) : value(v) {};

    /**
     * @deprecated replaced by a2c/c2a stream get_size methods
     *
     * Get the value of the size property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_size method")
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        value_type ret;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_prop_get_size(props, &ret) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * @deprecated replaced by a2c/c2a stream get_size methods
     *
     * Get the value of the size property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_size method")
    static result<value_type>
    get(enyx_hw_filter *filter)
    {
        value_type ret;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_filter_get_size(filter, &ret) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {ret};
    }

    /**
     * @deprecated replaced by a2c/c2a stream get_size methods
     *
     * Set the value of the size property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.0.0
     */
    ENYX_HW_CXX_DEPRECATED("replaced by a2c/c2a stream get_size method")
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
        if (enyx_hw_filter_set_size(filter, value) != 0)
#pragma GCC diagnostic pop
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.0.0
     */
    const value_type value;
};

/**
 * Filter resources using their type
 *
 * @since 5.9.5
 */
struct driver
{
    /**
     * The possible drivers.
     *
     * @since 5.9.5
     */
    enum class value_type {
        HFP = ENYX_HW_HFP_DRIVER,
        MOCK = ENYX_HW_MOCK_DRIVER
    };

    /**
     * Construct a driver property.
     *
     * @param v The value of the driver property.
     * @since 5.9.5
     */
    explicit
    driver(value_type const & v) : value(v) {};

    /**
     * Get the value of the driver property from properties
     *
     * @param props The properties to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.9.5
     */
    static result<value_type>
    get(enyx_hw_properties const *props)
    {
        int ret;
        if (enyx_hw_prop_get_driver(props, &ret) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {static_cast<value_type>(ret)};
    }

    /**
     * Get the value of the driver property from filter
     *
     * @param filter The filter to query.
     * @return A result containing either the value or an error.
     *
     * @since 5.9.5
     */
    static
    result<value_type> get(enyx_hw_filter *filter)
    {
        int ret;
        if (enyx_hw_filter_get_driver(filter, &ret) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {static_cast<value_type>(ret)};
    }

    /**
     * Set the value of the driver property for filter
     *
     * @param filter The filter to query.
     * @param value The value to set
     * @return A result containing nothing or an error.
     *
     * @since 5.9.5
     */
    static result<void>
    set(enyx_hw_filter *filter, value_type const &value)
    {
        if (enyx_hw_filter_set_driver(filter, static_cast<int>(value)) != 0)
            return {std::error_code{errno, std::generic_category()}};
        return {};
    }

    /**
     * A value used to set the property.
     *
     * @since 5.9.5
     */
    value_type value;
};


ENYX_HW_NAMESPACE_END

#include <enyx/hw/properties.ipp>
