/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator MM accesses from the CPU.
 */
#pragma once

#include <cstdint>
#include <system_error>
#include <memory>

#include <enyx/hw/fwd.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/properties.hpp>
#include <enyx/hw/result.hpp>
#include <enyx/hw/accelerator.hpp>

#include <enyx/hw_c/mmio.h>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_mmio
 */
class mmio final
{
public:
    /// The type used to store an MMIO address.
    using addr_type = std::uint64_t;

public:
    /**
     * Construct a mmio from an associated descriptor
     *
     * @param descriptor The descriptor to use
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    explicit
    mmio(mmio_descriptor const& descriptor);

    /**
     * Get the descriptor associated with this mmio.
     *
     * @return the descriptor.
     *
     * @since 5.0.0
     */
    mmio_descriptor
    get_descriptor() const;

    /**
     * Write the @p value at @p addr.
     *
     * @param addr The address of the write (in bytes).
     * @param value The integral value to write.
     * @returns A result that contains an error on failure and may be used
     *          to throw an exception.
     *
     * @since 5.0.0
     */
    result<void>
    write_8(addr_type addr, std::uint8_t value);

    /**
     * @copydoc write_8
     */
    result<void>
    write_16(addr_type addr, std::uint16_t value);

    /**
     * @copydoc write_8
     */
    result<void>
    write_32(addr_type addr, std::uint32_t value);

    /**
     * @copydoc write_8
     */
    result<void>
    write_64(addr_type addr, std::uint64_t value);

    /**
     * Read byte(s) at @p addr
     *
     * @param addr The address of the read (in bytes).
     * @returns A result object that contains the read value on success or an
     *          error on failure. It may be used to throw an exception.
     *
     * @since 5.0.0
     */
    result<std::uint8_t>
    read_8(addr_type addr);

    /**
     * @copydoc read_8
     */
    result<std::uint16_t>
    read_16(addr_type addr);

    /**
     * @copydoc read_8
     */
    result<std::uint32_t>
    read_32(addr_type addr);

    /**
     * @copydoc read_8
     */
    result<std::uint64_t>
    read_64(addr_type addr);

    /**
     * Direct access to the underlying C mmio object.
     *
     * @return The C mmio object.
     *
     * @since 5.0.0
     */
    enyx_hw_mmio * handle() noexcept;

    /**
     * Direct access to the underlying C mmio object.
     *
     * @return The C mmio object.
     *
     * @since 5.0.0
     */
    enyx_hw_mmio const * handle() const noexcept;

private:
    accelerator accelerator_;
    std::shared_ptr<enyx_hw_mmio> mmio_;
};

ENYX_HW_NAMESPACE_END

#include <enyx/hw/mmio.ipp>
