/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Core discovery.
 */
#pragma once

#include <enyx/hw_c/core.h>

#include <cerrno>
#include <cstddef>
#include <iostream>
#include <memory>
#include <stack>
#include <stack>
#include <string>
#include <system_error>
#include <utility>
#include <vector>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/result.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * This class represents an hardware core tree.
 *
 * @since 5.0.0
 */
class core_tree
{
public:
    /// @copydoc core::iterator
    using iterator = core::iterator;

public:
    /**
     * Construct a core tree
     *
     * @param mmio The mmio this core_tree belongs to.
     * @param core_tree_ptr The C API core instance to wrap
     *
     * @since 5.0.0
     */
    core_tree(mmio const & mmio, ::enyx_hw_core_tree * const core_tree_ptr);

    /**
     * Deleted copy constructor
     *
     * @param old The instance to copy from
     *
     * @since 5.0.0
     */
    core_tree(core_tree const& old) = delete;

    /**
     * Move constructor
     *
     * @since 5.0.0
     */
    core_tree(core_tree &&) noexcept = default;

    /**
     * Deleted move Assignment operator
     *
     * @param old The instance to copy from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    core_tree &
    operator=(core_tree const& old) = delete;

    /**
     * Assignment operator
     *
     * @note The @p old instance is left uninitialized
     *
     * @param old The instance to steal from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    core_tree &
    operator=(core_tree && old) noexcept = default;

    /**
     * Get root core
     *
     * @return A core instance
     *
     * @since 5.0.0
     */
    core
    get_root() const noexcept;

    /**
     *  Retrieve an iterator to the begin of the tree (i.e. root node)
     *
     *  @return The iterator
     *
     *  @since 5.7.0
     */
    core::iterator
    begin() const noexcept;

    /**
     *  Retrieve an iterator to the past-the-end of the tree
     *
     *  @note This iterator should only be compared and never dereferenced
     *
     *  @return The iterator
     *
     *  @since 5.7.0
     */
    core::iterator
    end() const noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_tree object.
     *
     * @return The C enyx_hw_core_tree object.
     *
     * @since 5.0.0
     */
    ::enyx_hw_core_tree *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_tree object.
     *
     * @return The C enyx_hw_core_tree object.
     *
     * @since 5.0.0
     */
    ::enyx_hw_core_tree const *
    handle() const noexcept;

private:
    using core_tree_ptr = std::shared_ptr<enyx_hw_core_tree>;

private:
    mmio mmio_;
    core_tree_ptr tree_;
};

/**
 * Print the core tree @p ct into the output stream @p out
 *
 * @param out The output stream
 * @param ct The core tree to print
 * @return A reference to @p out
 *
 * @since 5.0.0
 */
std::ostream &
operator<<(std::ostream & out, core_tree const& ct);

/**
 * Retrieve all cores within the @p channel address space
 *
 * @note This function will create individual shared mutexes for all cores in
 * @b /dev/shm with permissions @b 666. If some users of a machine should not
 * have access to these mutexes, defensive mechanism like SELinux should be
 * enforced.
 * Additionally, if you are using a kernel >= 4.19, the shm will be subject to
 * an additional restriction (search protected_regular in @b man @b 5 @b proc.
 * It can be bypassed by discovering the cores for the first time as root or by
 * disabling it (@b protected_regular set to 0).
 *
 * @param channel The channel used to discover and access the cores
 * @return A core tree containing the whole core hierarchy
 *
 * @since 5.0.0
 */
core_tree
enumerate_cores(mmio & channel);

ENYX_HW_NAMESPACE_END

#include <enyx/hw/core_tree.ipp>
