#include "UdpRuntime.hpp"

#include <cerrno>

#include <algorithm>
#include <atomic>
#include <exception>
#include <functional>
#include <future>
#include <iostream>
#include <memory>
#include <numeric>
#include <sstream>
#include <thread>
#include <vector>

#include <enyx/hw/error.hpp>

#include <enyx/cores/namespace.hpp>

#include "../lib/DeviceDiscovery.hpp"
#include "../lib/SignalHandler.hpp"
#include "DataValidation.hpp"

ENYX_CORES_NAMESPACE_BEGIN

TxState
UdpRuntime::Session::get_tx_state()
{
    if (tx_state.load(std::memory_order_acquire) != CONFIGURED)
        return TxState::NOT_READY;
    return TxState::READY;
}

std::size_t
UdpRuntime::Session::get_payload_max_size() const
{
    // XXX: should use a random value from configuration->packet_size range
    return 2000;
}

UdpRuntime::UdpRuntime(ApplicationConfiguration const & configuration,
                       hw::accelerator & accelerator, hw::core_tree & core_tree)
    : data_sources(get_sources<UdpSource>(accelerator,
                                          configuration.stack_index,
                                          "UDP"))
    , data_sinks(get_sinks<UdpSink>(accelerator,
                                    configuration.stack_index,
                                    "UDP"))
    , stack(find_udp_core(core_tree, configuration.stack_index))
    , session_count(std::min(stack.get_tx_session_count(),
                             stack.get_rx_session_count()))
    , sessions(session_count)
    , open_sessions_count{0}
{
}

void
UdpRuntime::create_monitoring_tasks(Tasks & tasks)
{
}

void
UdpRuntime::connect_session(std::uint32_t session_id,
                                Session & session)
{
    auto const & configuration = *session.configuration;
    auto tx_session = stack.get_tx_session(session_id).value();
    auto rx_session = stack.get_rx_session(session_id).value();

    ipv4_endpoint endpoint{configuration.endpoint};

    udp::tx_session_parameters tx_parameters{};
    tx_parameters.interface_id = configuration.interface_id;
    tx_parameters.source_port = 0;
    tx_session.connect(endpoint.address(),
                       endpoint.port(),
                       tx_parameters).value();

    session.tx_state.store(Session::CONFIGURED, std::memory_order_release);

    std::uint16_t const source_port = tx_session.get_source_port().v();
    udp::rx_session_parameters rx_parameters{};
    rx_parameters.interface_id = configuration.interface_id;
    rx_parameters.source_port = source_port;
    rx_session.listen(rx_parameters).value();

    session.rx_state.store(Session::CONFIGURED, std::memory_order_release);
    std::cout << "session " << session_id << " rx configured" << std::endl;
    std::cout << "session " << session_id << " tx configured" << std::endl;
}

void
UdpRuntime::listen_on_session(std::uint32_t session_id,
                                  Session & session)
{
    auto const & configuration = *session.configuration;
    auto rx_session = stack.get_rx_session(session_id).value();
    udp::rx_session_parameters parameters{};
    parameters.interface_id = configuration.interface_id;
    parameters.source_port = std::stoi(configuration.endpoint);

    session.source_port.store(parameters.source_port,
                              std::memory_order_release);

    rx_session.listen(parameters).value();
    session.connect_on_receive = true;

    session.rx_state.store(Session::CONFIGURED, std::memory_order_release);
    std::cout << "session " << session_id << " rx configured" << std::endl;
}

void
UdpRuntime::multicast_listen_on_session(std::uint32_t session_id,
                                            Session & session)
{
    auto const & configuration = *session.configuration;
    auto rx_session = stack.get_rx_session(session_id).value();

    ipv4_endpoint endpoint{configuration.endpoint};

    udp::rx_session_parameters parameters{};
    parameters.interface_id = configuration.interface_id;
    parameters.source_port = endpoint.port();

    session.source_port.store(parameters.source_port,
                              std::memory_order_release);

    rx_session.listen_multicast(endpoint.address(), parameters).value();
    session.connect_on_receive = true;

    session.rx_state.store(Session::CONFIGURED, std::memory_order_release);
    std::cout << "session " << session_id << " rx configured" << std::endl;
}

void
UdpRuntime::open_session(std::uint32_t session_id,
                             Session & session)
{

    switch (session.configuration->type)
    {
    case SessionConfiguration::CLIENT:
        connect_session(session_id, session);
        break;

    case SessionConfiguration::SERVER:
        listen_on_session(session_id, session);
        break;

    case SessionConfiguration::MULTICAST:
        multicast_listen_on_session(session_id, session);
    }

    ++open_sessions_count;
}

void
UdpRuntime::close_session(std::uint32_t session_id)
{
    stack.get_rx_session(session_id).value().close().value();
    stack.get_tx_session(session_id).value().close().value();
    std::cout << "session " << session_id << " rx closed" << std::endl;
    std::cout << "session " << session_id << " tx closed" << std::endl;
    if (--open_sessions_count == 0)
        request_exit();
}

bool
UdpRuntime::is_tx_buffer_full(Session & session,
                              std::uint32_t size)
{
    return false;
}

ENYX_CORES_NAMESPACE_END
