#include "Application.hpp"

#include <enyx/hw/core_tree.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/tcp/session_monitor.hpp>

#include "ConfigurationParser.hpp"
#include "../lib/SignalHandler.hpp"
#include "../lib/DeviceDiscovery.hpp"

ENYX_CORES_NAMESPACE_BEGIN

Application::Application(ApplicationConfiguration const & configuration)
    : configuration_(configuration)
    , accelerator_(create_accelerator(configuration.accelerator))
    , mmio_(get_first_mmio(accelerator_))
    , core_tree_(hw::enumerate_cores(mmio_))
    , tcp_(find_tcp_core(core_tree_, configuration_.stack_index))
    , monitor_(tcp_)
    , open_session_count_(0)
{ }

void
Application::open_sessions()
{

    std::uint16_t session_id = 0;
    for (auto const& session_configuration: configuration_.session_configurations)
    {
        open_session(session_id++, session_configuration);
    }
}

void
Application::open_session(std::uint16_t session_id,
                          SessionConfiguration const & session_configuration)
{
    switch (session_configuration.type)
    {
    case SessionConfiguration::CLIENT:
        connect_session(session_id, session_configuration);
        break;

    case SessionConfiguration::SERVER:
        listen_on_session(session_id, session_configuration);
        break;
    }
    ++open_session_count_;
}

void
Application::connect_session(std::uint16_t session_id,
                             SessionConfiguration const & session_configuration)
{
    ipv4_endpoint endpoint{session_configuration.endpoint};

    tcp::session_parameters parameters{};
    parameters.interface_id = session_configuration.interface_id;
    auto tcp_session = tcp_.get_session(session_id).value();
    monitor_.add_session(tcp_session).value();
    tcp_session.connect(endpoint.address(), endpoint.port(), parameters)
               .value();
}

void
Application::listen_on_session(std::uint16_t session_id,
                               SessionConfiguration const & session_configuration)
{
    tcp::session_parameters parameters{};
    parameters.interface_id = session_configuration.interface_id;
    parameters.source_port = std::stoi(session_configuration.endpoint);

    auto tcp_session = tcp_.get_session(session_id).value();
    monitor_.add_session(tcp_session).value();

    tcp_session.listen(parameters).value();
}

void Application::run()
{
    std::cout << "Started." << std::endl;
    open_sessions();

    while (! is_exit_requested()) {
        auto events = monitor_.poll_events().value();
        for (auto const & event: events) {
            std::cout << "session " << event.session_id;
            switch (event.new_state) {
            case ENYX_TCP_SESSION_STATE_CLOSED:
                std::cout << " closed";
                if (--open_session_count_ == 0)
                    request_exit();
                break;
            case ENYX_TCP_SESSION_STATE_OPENING:
                std::cout << " half open";
                break;
            case ENYX_TCP_SESSION_STATE_ESTABLISHED:
                std::cout << " established";
                tcp_.get_session(event.session_id).value().close().value();
                break;
            case ENYX_TCP_SESSION_STATE_CLOSING:
                std::cout << " half closed";
                break;
            }
            std::cout << std::endl;
        }
    }
}

ENYX_CORES_NAMESPACE_END
