#include "FilterScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include <enyx/cores/hardware_ids.hpp>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace FilterScope {

namespace {

namespace en = enyx;
namespace ni = en::ethernet_filter;

ni::ethernet_filter
create_filter(hw::core & root,
           Application::StackType stack_type,
           std::uint32_t stack_index)
{
    std::vector<ni::ethernet_filter> ethernet_filter;

    if (stack_type == Application::StackType::ALL ||
            stack_type == Application::StackType::TCP ||
            stack_type == Application::StackType::TCP_ULL)
    {
        auto tcp_ull_cores = root.enumerate(enyx::hardware_ids::TCP_ULL);
        ethernet_filter.insert(ethernet_filter.end(),
                               tcp_ull_cores.begin(), tcp_ull_cores.end());
    }

    if (stack_type == Application::StackType::ALL ||
            stack_type == Application::StackType::UDP ||
            stack_type == Application::StackType::UDP_ULL)
    {
        auto udp_ull_cores = root.enumerate(enyx::hardware_ids::UDP_ULL);
        ethernet_filter.insert(ethernet_filter.end(),
                               udp_ull_cores.begin(), udp_ull_cores.end());
    }

    if (ethernet_filter.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(ethernet_filter[stack_index]);
}

char const USAGE[] =
"Usage: " BIN_NAME " ethernet_filter show\n"
"       " BIN_NAME " ethernet_filter set filtering ( enable | disable )\n"
"       " BIN_NAME " ethernet_filter set unicast_promiscuous ( enable | disable)\n"
"       " BIN_NAME " ethernet_filter set udp_session_copy ( enable | disable) UDP_SESSION_ID\n";

void
set_filtering_enable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    ensure_end(i, e);

    ethernet_filter.enable().value();
}

void
set_filtering_disable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    ensure_end(i, e);

    ethernet_filter.disable().value();
}

void
set_filtering(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected { enable | disable }"};

    if (is_prefix(*i, "enable"))
        set_filtering_enable(++i, e, ethernet_filter);
    else if (is_prefix(*i, "disable"))
        set_filtering_disable(++i, e, ethernet_filter);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
            << "', try '" BIN_NAME " ethernet_filter help'";
        throw std::runtime_error{error.str()};
    }
}

void
set_unicast_promiscuous_enable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    ensure_end(i, e);

    ethernet_filter.enable_unicast_promiscuous().value();
}

void
set_unicast_promiscuous_disable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    ensure_end(i, e);

    ethernet_filter.disable_unicast_promiscuous().value();
}

void
set_unicast_promiscuous(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected { enable | disable }"};

    if (is_prefix(*i, "enable"))
        set_unicast_promiscuous_enable(++i, e, ethernet_filter);
    else if (is_prefix(*i, "disable"))
        set_unicast_promiscuous_disable(++i, e, ethernet_filter);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
            << "', try '" BIN_NAME " ethernet_filter help'";
        throw std::runtime_error{error.str()};
    }
}

void
set_udp_session_copy_enable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected UDP session id"};

    auto const session = std::atoi(*i);
    ensure_end(++i, e);

    ethernet_filter.enable_udp_session_copy(session).value();
}

void
set_udp_session_copy_disable(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected UDP session id"};

    auto const session = std::atoi(*i);
    ensure_end(++i, e);

    ethernet_filter.disable_udp_session_copy(session).value();
}

void
set_udp_session_copy(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected { enable | disable }"};

    if (is_prefix(*i, "enable"))
        set_udp_session_copy_enable(++i, e, ethernet_filter);
    else if (is_prefix(*i, "disable"))
        set_udp_session_copy_disable(++i, e, ethernet_filter);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
            << "', try '" BIN_NAME " ethernet_filter help'";
        throw std::runtime_error{error.str()};
    }
}

void
set(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    if (i == e)
        throw std::runtime_error{"Expected server"};

    if (is_prefix(*i, "filtering"))
    {
        set_filtering(++i, e, ethernet_filter);
    }
    else if (is_prefix(*i, "unicast_promiscuous"))
    {
        set_unicast_promiscuous(++i, e, ethernet_filter);
    }
    else if (is_prefix(*i, "udp_session_copy"))
    {
        set_udp_session_copy(++i, e, ethernet_filter);
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " ethernet_filter help'";
        throw std::runtime_error{error.str()};
    }
}

void
show(CmdLineIterator i, CmdLineIterator e, ni::ethernet_filter & ethernet_filter)
{
    std::cout << "filtering: ";
    if (ethernet_filter.enabled())
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
    else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;
    std::cout << "\nunicast_promiscuous: ";
    if (ethernet_filter.unicast_promiscuous_enabled())
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
    else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;

    std::cout << "\nUDP session copy: ";
    uint16_t session = 0;
    while (auto res = ethernet_filter.udp_session_copy_enabled(session)) {
        if ((session % 4) == 0)
            std::cout << "\n";
        if (res.value())
            std::cout << "\t" << session << ": "
                      << Color::CONFIGURED << "enabled" << Color::NONE;
        else
            std::cout << "\t" << session << ": "
                      << Color::UNCONFIGURED << "disabled" << Color::NONE;
        session++;
    }
    if (session == 0)
        std::cout << Color::UNCONFIGURED << "unavailable" << Color::NONE;
    std::cout << std::endl;
}

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            ni::ethernet_filter & ethernet_filter)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, ethernet_filter);
    else if (is_prefix(*i, "set"))
        set(++i, e, ethernet_filter);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " ethernet_filter help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    ni::ethernet_filter ethernet_filter{create_filter(root, stack_type, stack_index)};
    parse_scope(i, e, ethernet_filter);
}

} // namespace FilterScope

ENYX_CORES_NAMESPACE_END
