#include <cstring>
#include <fstream>
#include <iostream>
#include <iterator>
#include <vector>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/core.hpp>

#include "../lib/DeviceDiscovery.hpp"

#include "runner.hpp"

namespace eh = enyx::hw;

runner::runner(std::string const & accelerator_arg)
    : accelerator_{enyx::create_accelerator(accelerator_arg)}
    , mmio_{enyx::get_first_mmio(accelerator_)}
    , core_tree_{eh::enumerate_cores(mmio_)}
    , root_{core_tree_.get_root()}
    , flash_(root_)
{ }

void
runner::run(struct run_params & params)
{
    params_ = params;

    switch (params_.command) {
    case flash_command::READ_REG:
        read_register();
        break;
    case flash_command::WRITE_REG:
        write_register();
        break;
    case flash_command::READ_MEM:
        read_memory();
        break;
    case flash_command::WRITE_MEM:
        write_memory();
        break;
    }
}

enyx::flash::flash::section const &
runner::get_flash_section(flash_section & section)
{
    switch (section) {
    case flash_section::PGM:
        return flash_.pgm();
    case flash_section::USR:
        return flash_.usr();
    case flash_section::PRM:
        return flash_.prm();
    case flash_section::OTP:
        return flash_.otp();
    }

    throw std::runtime_error{"Unsupported flash section"};
}

void
runner::read_memory()
{
    auto & section = get_flash_section(params_.target);
    std::vector<uint8_t> buffer(params_.size);

    auto res = section.read(buffer.data(), params_.offset, params_.size);
    if (! res) {
        throw std::runtime_error{"Failed to read data from flash memory"};
    }

    std::ofstream out_file{params_.file, std::ios::binary};

    std::copy(buffer.begin(), buffer.end(),
            std::ostream_iterator<std::uint8_t>(out_file));
}

void
runner::write_memory()
{
    auto & section = get_flash_section(params_.target);

    std::ifstream in_file{params_.file, std::ios::binary};
    in_file.unsetf(std::ios::skipws);
    auto buffer = std::vector<char>{
        std::istream_iterator<char>{in_file},
        std::istream_iterator<char>{}};

    if (buffer.size() < params_.size) {
        throw std::runtime_error{
                "Input file does not contains enough data to write"};
    }

    if (buffer.size() > params_.size) {
        std::cerr
            << "Input file contains more data than specified size. "
            << "Remaining data will not be written."
            << std::endl;
    }

    auto res = section.write(reinterpret_cast<uint8_t *>(buffer.data()),
            params_.offset, params_.size);
    if (! res) {
        throw std::runtime_error{"Failed to write data to flash memory"};
    }
}

void
runner::read_register()
{
    std::vector<uint8_t> buffer(params_.size);

    flash_.read_register(params_.reg_addr, buffer.data(), params_.size);

    for (auto byte: buffer) {
        std::cout << "0x" << std::hex << unsigned(byte) << " ";
    }
    std::cout << std::endl;
}

void
runner::write_register()
{
    uint8_t reg_value[4];
    std::memcpy(reg_value, &(params_.reg_value), sizeof(uint32_t));

    flash_.write_register(params_.reg_addr, reg_value, params_.size);
}
