#pragma once

#include <stdint.h>
#include <unistd.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/tcp/session.h>
#include <enyx/cores_c/tcp/tcp.h>

/**
 * Session monitor object.
 *
 * This object is used to monitor tcp sessions and detect status changes.
 *
 * Usage:
 * @code{.c}
 * int result = -1;
 * enyx_tcp_session_monitor * monitor;
 * enyx_tcp_session session;
 * enyx_tcp_session_parameters parameters;
 * // In this example, we only monitor 1 session.
 * enyx_tcp_session_monitor_event events[1];
 * ssize_t event_count;
 *
 * monitor = enyx_tcp_session_monitor_create(tcp);
 * if (! monitor) {
 *     perror("enyx_tcp_session_monitor_create");
 *     exit(1);
 * }
 *
 * if (enyx_tcp_get_available_session(tcp, &session) < 0) {
 *     perror("enyx_tcp_get_available_session");
 *     goto get_available_session_failed;
 * }
 *
 * if (enyx_tcp_session_monitor_add_session(monitor, &session) < 0) {
 *     perror("enyx_tcp_session_monitor_add_session");
 *     goto add_session_failed;
 * }
 *
 * if (enyx_tcp_session_connect(&session, address, port, &parameters) < 0) {
 *     perror("enyx_tcp_session_connect");
 *     goto connect_failed;
 * }
 *
 * sleep(5);
 *
 * event_count = enyx_tcp_session_monitor_poll_events(monitor, events, 1);
 * if (event_count < 0) {
 *     perrror("poll_events");
 *     goto poll_failed;
 * }
 *
 * if (event_count == 0) {
 *     printf("Session %"PRIu16" did not open\n", session.session_id);
 *     goto no_opening;
 * }
 *
 * switch (events[0].new_state) {
 * case ENYX_TCP_SESSION_STATE_OPENING:
 *     printf("Session %"PRIu16" is connecting\n", events[0].session_id);
 *     break;
 * case ENYX_TCP_SESSION_STATE_ESTABLISHED:
 *     printf("Session %"PRIu16" is connected\n", events[0].session_id);
 *     break;
 * default:
 *     printf("Unexpected status CLOSING for session %"PRIu16"\n",
 *            events[0].session_id);
 *     break;
 * };
 *
 * if (enyx_tcp_session_close(&session) < 0) {
 *     perror("enyx_tcp_session_close");
 *     goto close_failed;
 * }
 *
 * sleep(5);
 *
 * event_count = enyx_tcp_session_monitor_poll_events(monitor, events, 1);
 * if (event_count < 0) {
 *     perror("poll_events");
 *     goto poll_close_failed;
 * }
 *
 * if (event_count == 0) {
 *     printf("Session %"PRIu16" did not close\n", session.session_id);
 *     goto no_closing;
 * }
 *
 * switch (events[0].new_state) {
 * case ENYX_TCP_SESSION_STATE_CLOSING:
 *     printf("Session %"PRIu16" is closing\n", events[0].session_id);
 *     break;
 * case ENYX_TCP_SESSION_STATE_CLOSED:
 *     printf("Session %"PRIu16" is closed\n", events[0].session_id);
 *     break;
 * default:
 *     printf("Unexpected status for session %"PRIu16"\n",
 *            events[0].session_id);
 *     break;
 * };
 *
 * ret = 0;
 *
 * no_closing:
 * poll_close_failed:
 * close_failed:
 * no_opening:
 * poll_failed:
 * enyx_tcp_session_reset(&session);
 * connect_failed:
 * enyx_tcp_session_monitor_remove_session(&session);
 * add_session_failed:
 * get_available_session_failed:
 * enyx_tcp_session_monitor_destroy(monitor);
 * return ret;
 * @endcode
 */
typedef struct enyx_tcp_session_monitor enyx_tcp_session_monitor;

/**
 * Create a monitor from a tcp subsystem
 *
 * @param tcp The tcp subsystem.
 * @return The monitor on success. NULL on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_tcp_session_monitor *
enyx_tcp_session_monitor_create(enyx_tcp const * tcp);

/**
 * Destroy a monitor.
 *
 * @param monitor The monitor to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_tcp_session_monitor_destroy(enyx_tcp_session_monitor * monitor);

/**
 * Add a session to be watched.
 *
 * @param monitor The monitor
 * @param session The session.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_monitor_add_session(enyx_tcp_session_monitor * monitor,
                                     enyx_tcp_session const * session);

/**
 * Remove a session from watched sessions.
 *
 * @param monitor The monitor
 * @param session The session.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_monitor_remove_session(enyx_tcp_session_monitor * monitor,
                                        enyx_tcp_session const * session);

/**
 * Get the number of monitored sessions.
 *
 * @param monitor The monitor
 * @return The number of monitored session.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY size_t
enyx_tcp_session_monitor_count(enyx_tcp_session_monitor const * monitor);

/**
 * An event as returned by enx_session_monitor_poll_events.
 */
typedef struct {
    uint16_t session_id;
    enyx_tcp_session_state new_state;
} enyx_tcp_session_monitor_event;

/**
 * Poll all monitored sessions.
 *
 * For each session whose status has changed since last poll, an event will be
 * created.
 *
 * @note This function does a lot of MM access, you may want to wait a bit
 * between each calls.
 *
 * @param monitor The monitor used for polling.
 * @param events The returned events.
 * @param size Size of events
 *
 * @return The number of events on success. -1 on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_tcp_session_monitor_poll_events(enyx_tcp_session_monitor const * monitor,
                                     enyx_tcp_session_monitor_event * events,
                                     size_t size);
