/// @file icmp.h
#pragma once

#include <stdbool.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>

/**
 * ICMP subsystem object.
 *
 * This object represents the ICMP subsystem of a hardware network stack.
 *
 * Usage:
 * @code{.c}
 * int result = -1;
 * enyx_icmp * icmp;
 *
 * // Create the ICMP object
 * icmp = enyx_icmp_create(tcp_core);
 * if (! icmp)
 * {
 *     perror("enyx_icmp_create");
 *     goto create_failed;
 * }
 *
 * // Get and print the current configuration of the ICMP subsystem
 * printf("ICMP server is %s\n",
 *        enyx_icmp_enabled(icmp) ? "enabled" : "disabled");
 *
 * // Disable ICMP server
 * if (enyx_icmp_disable(icmp) < 0)
 * {
 *     perror("enyx_icmp_disable");
 *     goto disable_failed:
 * }
 *
 * result = 0;
 * disable_failed:
 * enyx_icmp_destroy(icmp);
 * create_failed:
 * return result;
 * @endcode
 */
typedef struct enyx_icmp enyx_icmp;

/**
 * Construct the ICMP module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 * * tcp_multi_stack (from TCP_STD 2.x)
 * * udp_multi_stack (from UDP_STD 2.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible hw::core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_icmp *
enyx_icmp_create(enyx_hw_core * stack);

/**
 * Destroy an ICMP subsystem.
 *
 * @param icmp The ICMP subsystem to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_icmp_destroy(enyx_icmp * icmp);

/**
 * Check if the ICMP server is enabled.
 *
 * @param icmp The ICMP subsystem.
 * @return true if the ICMP server is enabled.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_icmp_enabled(enyx_icmp const * icmp);

/**
 * Enable the ICMP server of this subsystem.
 *
 * @param icmp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_icmp_enable(enyx_icmp * icmp);

/**
 * Disable the ICMP server of this subsystem.
 *
 * @param icmp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_icmp_disable(enyx_icmp * icmp);
