#include <stdexcept>
#include <system_error>

ENYX_CORES_NAMESPACE_BEGIN

namespace transceiver {

inline
sfp::sfp(std::shared_ptr<::enyx_transceiver> const & transceiver,
         ::enyx_sfp * sfp)
    : transceiver_(transceiver)
    , sfp_(sfp)
{
    if (! sfp_)
        throw std::runtime_error{"Cannot create SFP: SFP pointer is null"};
}

inline result<uint8_t>
sfp::get_id() const
{
    auto id = enyx_sfp_get_id(handle());
    if (id < 0)
        return std::error_code{errno, std::generic_category()};

    return id;
}

inline result<bool>
sfp::is_transmitter_enabled() const
{
    bool enabled;
    if (enyx_sfp_is_transmitter_enabled(handle(), &enabled) < 0)
        return std::error_code{errno, std::generic_category()};

    return enabled;
}

inline result<void>
sfp::enable_transmitter()
{
    if (enyx_sfp_enable_transmitter(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
sfp::disable_transmitter()
{
    if (enyx_sfp_disable_transmitter(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<bool>
sfp::get_tx_fault() const
{
    bool tx_fault;
    if (enyx_sfp_get_tx_fault(handle(), &tx_fault) < 0)
        return std::error_code{errno, std::generic_category()};

    return tx_fault;
}

inline result<sfp::sfp_rate>
sfp::get_rate() const
{
    sfp_rate rate;
    if (enyx_sfp_get_rate(handle(), &rate) < 0)
        return std::error_code{errno, std::generic_category()};

    return rate;
}

inline result<void>
sfp::set_rate(sfp::sfp_rate rate)
{
    if (enyx_sfp_set_rate(handle(), rate) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<bool>
sfp::has_signal() const
{
    bool signal;
    if (enyx_sfp_has_signal(handle(), &signal) < 0)
        return std::error_code{errno, std::generic_category()};

    return signal;
}

inline ::enyx_sfp *
sfp::handle()
{
    return sfp_.get();
}

inline ::enyx_sfp const *
sfp::handle() const
{
    return sfp_.get();
}

inline
qsfp::qsfp(std::shared_ptr<enyx_transceiver> const & transceiver,
           ::enyx_qsfp * qsfp)
    : transceiver_(transceiver)
    , qsfp_(qsfp)
{
    if (! qsfp_)
        throw std::runtime_error{"Cannot create QSFP: QSFP pointer is null"};
}

inline result<uint8_t>
qsfp::get_id() const
{
    auto id = enyx_qsfp_get_id(handle());
    if (id < 0)
        return std::error_code{errno, std::generic_category()};

    return id;
}

inline result<bool>
qsfp::is_low_power_mode_enabled()
{
    bool enabled;
    if (enyx_qsfp_is_low_power_mode_enabled(handle(), &enabled) < 0)
        return std::error_code{errno, std::generic_category()};

    return enabled;
}

inline result<void>
qsfp::enable_low_power_mode()
{
    if (enyx_qsfp_enable_low_power_mode(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
qsfp::disable_low_power_mode()
{
    if (enyx_qsfp_disable_low_power_mode(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<bool>
qsfp::is_i2c_communication_enabled()
{
    bool enabled;
    if (enyx_qsfp_is_i2c_communication_enabled(handle(), &enabled) < 0)
        return std::error_code{errno, std::generic_category()};

    return enabled;
}

inline result<void>
qsfp::enable_i2c_communication()
{
    if (enyx_qsfp_enable_i2c_communication(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
qsfp::disable_i2c_communication()
{
    if (enyx_qsfp_disable_i2c_communication(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
qsfp::reset_module()
{
    if (enyx_qsfp_reset_module(handle()) < 0)
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<bool>
qsfp::possible_fault_detected() const
{
    bool fault;
    if (enyx_qsfp_possible_fault_detected(handle(), &fault) < 0)
        return std::error_code{errno, std::generic_category()};

    return fault;
}

inline ::enyx_qsfp *
qsfp::handle()
{
    return qsfp_.get();
}

inline ::enyx_qsfp const *
qsfp::handle() const
{
    return qsfp_.get();
}

inline
transceiver::transceiver(hw::core & root)
    : transceiver_(::enyx_transceiver_create(root.handle()),
                   ::enyx_transceiver_destroy)
{
    if (! transceiver_)
        throw std::runtime_error{"Cannot create transceiver subsystem"};
}

inline result<size_t>
transceiver::count_sfp() const
{
    auto count = enyx_transceiver_count_sfp(handle());
    if (count < 0)
        return std::error_code{errno, std::generic_category()};

    return count;
}

inline result<size_t>
transceiver::count_qsfp() const
{
    auto count = enyx_transceiver_count_qsfp(handle());
    if (count < 0)
        return std::error_code{errno, std::generic_category()};

    return count;
}

inline result<sfp>
transceiver::get_sfp(uint8_t port)
{
    auto sfp = enyx_transceiver_get_sfp(handle(), port);
    if (! sfp)
        return std::error_code{errno, std::generic_category()};

    return {transceiver_, sfp};
}

inline result<qsfp>
transceiver::get_qsfp(uint8_t port)
{
    auto qsfp = enyx_transceiver_get_qsfp(handle(), port);
    if (! qsfp)
        return std::error_code{errno, std::generic_category()};

    return {transceiver_, qsfp};
}

inline ::enyx_transceiver *
transceiver::handle()
{
    return transceiver_.get();
}

inline ::enyx_transceiver const *
transceiver::handle() const
{
    return transceiver_.get();
}

} /* namespace transceiver */

ENYX_CORES_NAMESPACE_END
