#pragma once

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores_c/test_point/test_point.h>
#include <enyx/hw/core.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_test_point>
{
    void
    operator()(::enyx_test_point * ptr) const
    {
        ::enyx_test_point_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace test_point {

/**
 * @copydoc enyx_test_point
 */
class test_point {
public:
    /**
     * @copybrief enyx_test_point_create
     *
     * @param core The avl_st_test_point HW core.
     */
    test_point(enyx::hw::core const & core);

    /**
     * @copybrief enyx_test_point_get_packet_count
     *
     * @return A result containing the packet count or an error.
     */
    result<uint32_t>
    get_packet_count() const noexcept;

    /**
     * @copybrief enyx_test_point_get_bytes_count
     *
     * @return A result containing the byte count or an error.
     */
    result<uint32_t>
    get_bytes_count() const noexcept;

    /**
     * @copybrief enyx_test_point_get_error_count
     *
     * @return A result containing the error count or an error.
     */
    result<uint32_t>
    get_error_count() const noexcept;

    /**
     * @copybrief enyx_test_point_get_drop_count
     *
     * @return A result containing the drop count or an error.
     */
    result<uint32_t>
    get_drop_count() const noexcept;

    /**
     * @copybrief enyx_test_point_is_drop_mode_enabled
     *
     * @return A result containing the drop mode activation or an error.
     */
    result<bool>
    is_drop_mode_enabled() const noexcept;

    /**
     * @copybrief enyx_test_point_enable_drop_mode
     *
     * @return A result containing an error on failure.
     */
    result<void>
    enable_drop_mode() const noexcept;

    /**
     * @copybrief enyx_test_point_disable_drop_mode
     *
     * @return A result containing an error on failure.
     */
    result<void>
    disable_drop_mode() const noexcept;

    /**
     * @copybrief enyx_test_point_get_link_status
     *
     * @return A result containing the link status or an error.
     */
    result<bool>
    get_link_status() const noexcept;

    /**
     * @copybrief enyx_test_point_get_link_loss_count
     *
     * @return A result containing the link loss count or an error.
     */
    result<uint32_t>
    get_link_loss_count() const noexcept;

    /**
     * @copybrief enyx_test_point_is_ready_to_drop
     *
     * @return A result indicating if the test point is ready to drop or an
     * error.
     */
    result<bool>
    is_ready_to_drop() const noexcept;

    /**
     * @copybrief enyx_test_point_is_ready_to_backpressure
     *
     * @return A result indicating if the test point is ready to backpressure
     * or an error.
     */
    result<bool>
    is_ready_to_backpressure() const noexcept;

    /**
     * @copybrief enyx_test_is_ready_to_generate_errors
     *
     * @return A result indicating if the test point is ready to generate
     * errors or an error.
     */
    result<bool>
    is_ready_to_generate_errors() const noexcept;

    /**
     * @copybrief enyx_test_point_generate_drops
     *
     * @param count The number of packet to drop.
     * @return A result containing an error on failure.
     */
    result<void>
    generate_drops(uint16_t count) noexcept;

    /**
     * @copybrief enyx_test_point_generate_backpressure
     *
     * @param ticks The duration of backpressure in ticks.
     * @return A result containing an error on failure.
     */
    result<void>
    generate_backpressure(uint16_t ticks) noexcept;

    /**
     * @copybrief enyx_test_point_generate_errors
     *
     * @param count The number of packet to set in error.
     * @return A result containing an error on failure.
     */
    result<void>
    generate_errors(uint16_t count) noexcept;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_test_point.
     *
     * @return A pointer to a @ref enyx_test_point.
     */
    ::enyx_test_point *
    handle() noexcept;

    /**
     * Get a const pointer to the underlying @b C handle @ref enyx_test_point.
     *
     * @return A const pointer to a @ref enyx_test_point.
     */
    ::enyx_test_point const *
    handle() const noexcept;

private:
    enyx::hw::core test_point_core_;
    std::unique_ptr<::enyx_test_point> test_point_c_;
};

} /* namespace test_point */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/test_point/test_point.ipp>
