#pragma once

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/tcp/session.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores_c/tcp/tcp.h>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_tcp>
{
    void
    operator()(::enyx_tcp * ptr) const
    {
        ::enyx_tcp_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

/// @copydoc enyx_tcp_statistics
using statistics = enyx_tcp_statistics;

/**
 * @copydoc enyx_tcp
 */
class tcp
{
public:
    /**
     * Construct the TCP module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * tcp_multi_stack (from TCP_STD 2.x)
     *
     * @param tcp_core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    tcp(enyx::hw::core const& tcp_core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Get the name of this stack.
     *
     * @return A result object containing the stack name on success or an
     *         error on failure.
     */
    result<std::string>
    get_name() const noexcept;

    /**
     * Test if this TCP subsystem has instant ack.
     *
     * @return true if instant ack is enabled.
     */
    bool
    has_instant_ack() const noexcept;

    /**
     * Get the maximum segment size supported by this TCP subsystem.
     *
     * @return A result object containing the MSS on success or an error on
     *         failure.
     */
    result<std::uint16_t>
    get_mss() const noexcept;

    /**
     * Is safe mode enabled on this tcp subsystem.
     *
     * @note If the underlying hardware core is a tcp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return true if safe mode is enabled, false otherwise.
     */
    bool
    is_safe_mode_enabled() const noexcept;

    /**
     * Enable safe mode on this subsystem.
     *
     * @note If the underlying hardware core is a tcp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_safe_mode() noexcept;

    /**
     * Disable safe mode on this subsystem.
     *
     * @note If the underlying hardware core is a tcp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_safe_mode() noexcept;

    /**
     * @copybrief enyx_tcp_is_checksum_verification_enabled
     *
     * @return true if checksum verification is enabled, false otherwise
     */
    bool
    is_checksum_verification_enabled() const noexcept;

    /**
     * @copybrief enyx_tcp_enable_checksum_verification
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_checksum_verification();

    /**
     * @copybrief enyx_tcp_disable_checksum_verification
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_checksum_verification();

    /**
     * @copybrief enyx_tcp_get_syn_retry_timeout
     *
     * @return A result object containing the timeout on success or an error on
     *         failure.
     */
    result<std::uint8_t>
    get_syn_retry_timeout() const noexcept;

    /**
     * @copybrief enyx_tcp_set_syn_retry_timeout
     *
     * @param retry_timeout_ms The timeout to set
     * @return A result object containing an error on failure.
     *
     * @warning retry_timeout_s maximum value is 0xF (4 bits).
     */
    result<void>
    set_syn_retry_timeout(std::uint8_t retry_timeout_ms);

    /**
     * @copybrief enyx_tcp_is_passive_close_enabled
     *
     * @return true if passive close is enabled, false otherwise
     */
    bool
    is_passive_close_enabled() const noexcept;

    /**
     * @copybrief enyx_tcp_enable_passive_close
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_passive_close();

    /**
     * @copybrief enyx_tcp_disable_passive_close
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_passive_close();

    /**
     * @copybrief enyx_tcp_get_maximum_segment_lifetime
     *
     * @return A result object containing the lifetime on success an error on
     *         failure.
     */
    result<std::uint32_t>
    get_maximum_segment_lifetime() const noexcept;

    /**
     * @copybrief enyx_tcp_set_maximum_segment_lifetime
     *
     * @param lifetime_ms The lifetime to set
     * @return A result object containing an error on failure.
     */
    result<void>
    set_maximum_segment_lifetime(std::uint32_t lifetime_ms);

    /**
     * Get the size of the RX fifo
     *
     * @return A result object containing fifo size on success or an error on
     * failure
     */
    result<std::uint8_t>
    get_rx_fifo_size() const noexcept;

    /**
     * Check if the stack will drop outgoing packets on sessions that are not
     * established
     *
     * @return true if enabled
     */
    bool
    drop_tx_packets_if_not_established_enabled() const noexcept;

    /**
     * Check if the OOS_SEQNUM feature is enabled.
     *
     * @return true if enabled
     */
    bool
    oos_seqnum_enabled() const noexcept;

    /**
     * Check if status events are sent to the software.
     *
     * @return true if enabled
     */
    bool
    status_events_enabled() const noexcept;

    /**
     * Check if TX Credit events are sent to the software.
     *
     * @return true if enabled
     */
    bool
    credit_events_enabled() const noexcept;

    /**
     * Check if retransmission External Memory is enabled.
     *
     * @return true if enabled
     */
    bool
    retransmission_external_memory_enabled() const noexcept;

    /**
     * Check if retransmission external full duplex memory is enabled.
     *
     * @return true if enabled
     */
    bool
    retransmission_full_duplex_external_memory_enabled() const noexcept;

    /**
     * Get retransmission external memory read latency.
     *
     * @return A result object containing the latency on success or an error on
     * failure
     */
    result<std::uint8_t>
    get_retransmission_external_memory_latency() const noexcept;

    /**
     * Get retransmission external/internal memory address width.
     *
     * @return A result object containing the address width on success or an
     * error on failure
     */
    result<std::uint8_t>
    get_retransmission_memory_address_width() const noexcept;

    /**
     * Get the retransmission external/internal memory data width.
     *
     * @return A result object containing the retransmission data width on
     * success or an error on failure
     */
    result<std::uint16_t>
    get_retransmission_memory_data_width() const noexcept;

    /**
     * Check if reordering is enabled.
     *
     * @return true if enabled
     */
    bool
    reordering_enabled() const noexcept;

    /**
     * Get the reordering memory address width.
     *
     * @return A result object containing the reordering address width on
     * success or an error on failure
     */
    result<std::uint8_t>
    get_reordering_memory_address_width() const noexcept;

    /**
     * Get the reordering memory data width.
     *
     * @return A result object containing the reordering memory data width on
     * success or an error on failure
     */
    result<std::uint16_t>
    get_reordering_memory_data_width() const noexcept;

    /**
     * Get the number of sessions supported by this TCP subsystem.
     *
     * @return The number of sessions.
     */
    std::uint16_t
    get_session_count() const noexcept;

    /**
     * Get a session object from this TCP subsystem.
     *
     * @param session_id The id of the requested session.
     *
     * @return A result object containing the session on success or an error
     *         on failure.
     */
    result<session>
    get_session(std::uint16_t session_id) const noexcept;

    /**
     * @copybrief enyx_tcp_get_available_session
     *
     * @return A result object containing a session on success or an error
     *         on failure.
     */
    result<session>
    get_available_session() const noexcept;

    /**
     * @copybrief enyx_tcp_get_statistics
     *
     * @return A result object containing the statistics on success or an error
     *         on failure.
     */
    result<statistics>
    get_statistics() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_tcp *
    handle() const noexcept;

private:
    enyx::hw::core tcp_core_;
    std::shared_ptr<::enyx_tcp> tcp_;
};

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/tcp/tcp.ipp>
