#pragma once

#include <enyx/cores_c/raw_ethernet/raw_ethernet.h>

#include <enyx/cores/result.hpp>

#include <enyx/cores/namespace.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_raw_ethernet>
{
    void
    operator()(::enyx_raw_ethernet * ptr) const
    {
        ::enyx_raw_ethernet_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace raw_ethernet {

class raw_ethernet
{
public:
    /**
     * Construct the RAW ETHERNET module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     *
     * @param core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    raw_ethernet(enyx::hw::core const& core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Get the MAC data width.
     *
     * @return A result object containing the mac data width on success or an
     * error on failure
     */
    enyx::result<std::uint8_t>
    get_mac_data_width() const noexcept;

    /**
     * Get the raw_ethernet data width.
     *
     * @return A result object containing the data width on success or an error
     * on failure
     */
    enyx::result<std::uint8_t>
    get_data_width() const noexcept;

    /**
     * Check is the raw_ethernet capture mode is enabled.
     *
     * @return The raw_ethernet capture mode activation status
     */
    bool
    capture_enabled() const noexcept;
    result<void>

    /**
     * Enable the raw_ethernet capture mode
     *
     * @return A result object containing an error on failure.
     */
    enable_capture() noexcept;

    /**
     * Disable the raw_ethernet capture mode
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_capture() noexcept;

    /**
     * Check if the raw_ethernet RX net_interface is available.
     *
     * @return A result object containing the raw_ethernet RX net_interface availability on
     * success or an error on failure
     */
    bool
    rx_interface_available() const noexcept;
    enyx::result<std::uint32_t>

    /**
     * Get the raw_ethernet RX total packet count.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    get_rx_packet_count() const noexcept;

    /**
     * Get the raw_ethernet RX packet count in error state.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_rx_error_count() const noexcept;

    /**
     * Check if the raw_ethernet TX net_interface is available.
     *
     * @return A result object containing the raw_ethernet TX net_interface availability on
     * success or an error on failure
     */
    bool
    tx_interface_available() const noexcept;

    /**
     * Get the raw_ethernet TX total packet count.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_tx_packet_count() const noexcept;

    /**
     * Get the raw_ethernet TX packet count in error state.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_tx_error_count() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_raw_ethernet *
    handle() const noexcept;

private:
    enyx::hw::core raw_ethernet_core_;
    std::unique_ptr<::enyx_raw_ethernet> raw_ethernet_;
};

} /* namespace raw_ethernet */
ENYX_CORES_NAMESPACE_END

#include "raw_ethernet.ipp"
