/**
 *  @file
 *
 *  Contains the types and functions related to the Event collector.
 */
#pragma once

#include <enyx/cores_c/probes/collector.h>

#include <cerrno>
#include <iostream>
#include <vector>
#include <algorithm>
#include <system_error>

#include <enyx/hw/result.hpp>
#include <enyx/hw/mmio.hpp>

#include <enyx/cores/namespace.hpp>

#include <enyx/cores/probes/probe.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_probes_collector>
{
    void
    operator()(::enyx_probes_collector * ptr) const
    {
        ::enyx_probes_collector_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/**
 * @copydoc enyx_probes_collector
 *
 * @since 6.7.0
 */
class collector final
{
/// @cond
public:
    friend std::ostream &
    operator<<(std::ostream & out, collector const& p);
/// @endcond

public:
    /**
     * Construct an event handler from a HW event collector core.
     *
     * @param core The event collector HW core.
     *
     * @since 6.7.0
     */
    collector(hw::core const & core);

    /**
     * Construct an event handler from its C handle
     *
     * @note This constructor should not be called directly.
     *
     * @param root The root core.
     * @param handle The collector handle to own.
     *
     * @since 6.7.0
     */
    collector(hw::core const & root, ::enyx_probes_collector * handle);

    /**
     * Direct access to the underlying C enyx_probes_collector object.
     *
     * @return The C enyx_probes_collector object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_collector *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_probes_collector object.
     *
     * @return The C enyx_probes_collector object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_collector const *
    handle() const noexcept;

    /**
     * Retrieve the collector name.
     *
     * @return the name string.
     *
     * @since 6.7.0
     */
    std::string
    get_name() const;

    /**
     * Retrieve the collector DMA stream name.
     *
     * @return the stream name string.
     *
     * @since 6.7.0
     */
    std::string
    get_stream_name() const;

private:
    hw::core core_;
    std::unique_ptr<::enyx_probes_collector> collector_;
};

/**
 * Print the collector @p m into the output stream @p out
 *
 * @param out The output stream
 * @param p The collector to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, collector const& p);

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/collector.ipp>
