#pragma once

#include <enyx/cores_c/icmp/icmp.h>

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_icmp>
{
    void
    operator()(::enyx_icmp * ptr) const
    {
        ::enyx_icmp_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace icmp {

/**
 * Manipulate hardware ICMP subsystem.
 *
 * This wrapper provides @b RAII of @b C @ref enyx_icmp
 */
class icmp
{
public:
    /**
     * Construct the ICMP module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     * * tcp_multi_stack (from TCP_STD 2.x)
     * * udp_multi_stack (from UDP_STD 2.x)
     *
     * @param core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    icmp(enyx::hw::core const& core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Check if the ICMP server is enabled.
     *
     * @return true if the ICMP server is enabled.
     */
    bool
    enabled() const noexcept;

    /**
     * Enable the ICMP server of this subsystem.
     *
     * @return A result object containg an error on failure
     */
    result<void>
    enable() noexcept;

    /**
     * Disable the ICMP server of this subsystem.
     *
     * @return A result object containg an error on failure
     */
    result<void>
    disable() noexcept;

    /**
     * Access to the C handle.
     *
     * @return a const pointer to the C handle
     */
    ::enyx_icmp const *
    handle() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return a pointer to the C handle
     */
    ::enyx_icmp *
    handle() noexcept;

private:
    enyx::hw::core icmp_core_;
    std::unique_ptr<::enyx_icmp> handle_;
};

} /* namespace icmp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/icmp/icmp.ipp>
