#pragma once

#include <enyx/cores_c/ethernet_flow_control/ethernet_flow_control.h>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/hw/core.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_ethernet_flow_control>
{
    void
    operator()(::enyx_ethernet_flow_control * ptr) const
    {
        ::enyx_ethernet_flow_control_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace ethernet_flow_control {

/**
 * @copybrief enyx_ethernet_flow_control
 *
 * This wrapper provides @b RAII of @b C @ref enyx_ethernet_flow_control
 *
 * Usage:
 *
 * @code{.cpp}
 * ethernet_flow_control flow_control{udp_stack};
 *
 * // Get the current ethernet flow control status
 * bool enabled = flow_control.is_enabled();
 * std::cout << "Ethernet flow control is " << enabled ? "enabled" : "disabled"
 *           << std::endl;
 *
 * // Set first virtual interface (ID 0) as ethernet flow control source interface
 * // Virtual interfaces IDs are the same as in the net_interface subsystem
 * auto set_source_res = flow_control.set_source_interface(0);
 * if (! set_source_res)
 *     throw std::system_error{set_source_res.error(),
 *                             "Cannot set source interface"};
 *
 * // Enable ethernet flow control
 * auto enable_res = flow_control.enable();
 * if (! enable_res)
 *     throw std::system_error{enable_res.error(),
 *                             "Cannot enable ethernet flow control"};
 *
 * return 0;
 * @endcode
 */
class ethernet_flow_control
{
public:
    /**
     * @copybrief enyx_ethernet_flow_control_create
     *
     * This module configures the following hw core:
     *
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     *
     * @param stack The hardware core containing the stack.
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    ethernet_flow_control(enyx::hw::core const& stack);

    /**
     * @copybrief enyx_ethernet_flow_control_is_enabled
     *
     * @return true if enabled, false otherwise.
     */
    bool
    is_enabled() const noexcept;

    /**
     * @copybrief enyx_ethernet_flow_control_enable
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable();

    /**
     * @copybrief enyx_ethernet_flow_control_disable
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable();

    /**
     * @copybrief enyx_ethernet_flow_control_get_buffer_size
     *
     * @return A result object containing the buffer size or an error on
     * failure.
     */
    result<std::uint32_t>
    get_buffer_size() const noexcept;

    /**
     * @copybrief enyx_ethernet_flow_control_get_internal_word_size
     *
     * @return A result object containing the internal word size or an error on
     * failure.
     */
    result<std::uint32_t>
    get_internal_word_size() const noexcept;

    /**
     * @copybrief enyx_ethernet_flow_control_get_used_buffer
     *
     * @return A result object containing the buffer usage or an error on
     * failure.
     */
    result<std::uint32_t>
    get_used_buffer() const noexcept;

    /**
     * @copybrief enyx_ethernet_flow_control_get_pause_threshold
     *
     * @return A result object containing the pause threshold or an error on
     * failure.
     */
    result<std::uint32_t>
    get_pause_threshold() const noexcept;

    /**
     * @brief Set the pause threshold.
     *
     * When the used buffer reach this @p threshold, ethernet flow control will
     * start sending ethernet pause frames until @b resume_threshold is
     * reached.
     *
     * Accepted values are [internal word size;buffer size[, given value will
     * be rounded down to the internal word size.
     * Internal word size is a property configured during synthesis.
     *
     * @param threshold The threshold in bytes.
     * @return A result object containing an error on failure.
     */
    result<void>
    set_pause_threshold(std::uint32_t threshold);

    /**
     * @copybrief enyx_ethernet_flow_control_get_resume_threshold
     *
     * @return A result object containing the resume threshold or an error on
     * failure.
     */
    result<std::uint32_t>
    get_resume_threshold() const noexcept;

    /**
     * @brief Set the resume threshold.
     *
     * When the used buffer reach this @p threshold, ethernet flow control will
     * stop sending ethernet pause frames until @b pause_threshold is reached
     * again.
     *
     * Accepted values are [internal word size;buffer size[, given value will
     * be rounded down to the internal word size.
     * Internal word size is a property configured during synthesis.
     *
     * @param threshold The threshold in bytes.
     * @return A result object containing an error on failure.
     */
    result<void>
    set_resume_threshold(std::uint32_t threshold);

    /**
     * @copybrief enyx_ethernet_flow_control_get_source_interface
     *
     * @return A result object containing the source interface ID or an
     * error on failure.
     */
    result<std::uint32_t>
    get_source_interface() const noexcept;

    /**
     * @copybrief enyx_ethernet_flow_control_set_source_interface
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    set_source_interface(std::uint32_t interface_id);

    /**
     * @copybrief enyx_ethernet_flow_control_get_pause_time
     *
     * Default: maximum pause time value 65535.
     *
     * @return A result object containing the pause time or an error on
     * failure.
     */
    result<std::uint32_t>
    get_pause_time() const noexcept;

    /**
     * @brief Set the pause time used in pause frames.
     *
     * This pause time will be used in ethernet pause frames sent when flow
     * control is occuring. It will inform the sender side to pause sending
     * data for this amount of time.
     *
     * The pause time is measured in units of pause "quanta", where each unit
     * is equal to (512 / net stack speed) seconds (ex. 512/10e9 for the 10G
     * ULL).
     * See https://en.wikipedia.org/wiki/Ethernet_flow_control#Pause_frame for
     * more information.
     *
     * Warning: unless for specific cases, default value should be kept.
     *
     * @param pause_time The pause time.
     * @return A result object containing an error on failure.
     */
    result<void>
    set_pause_time(std::uint16_t pause_time);

    /**
     * @copybrief enyx_ethernet_flow_control_get_pause_frame_count
     *
     * @return A result object containing the pause frame count or an
     * error on failure.
     */
    result<std::uint32_t>
    get_pause_frame_count() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return a const pointer to the C handle
     */
    ::enyx_ethernet_flow_control const *
    handle() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return a pointer to the C handle
     */
    ::enyx_ethernet_flow_control *
    handle() noexcept;

private:
    hw::core core_;
    std::unique_ptr<::enyx_ethernet_flow_control> handle_;
};

} // namespace ethernet_flow_control
ENYX_CORES_NAMESPACE_END

#include "ethernet_flow_control.ipp"
