#pragma once

#include <iterator>
#include <regex>
#include <vector>

#include <enyx/hw/properties.hpp>
#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core_tree.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/tcp/tcp.hpp>
#include <enyx/cores/udp/udp.hpp>

ENYX_CORES_NAMESPACE_BEGIN

hw::accelerator
create_accelerator(std::string const & accelerator);

hw::mmio
get_first_mmio(hw::accelerator & accelerator);

hw::core
find_tcp_core(hw::core_tree & core_tree, std::uint32_t stack_index);

hw::core
find_udp_core(hw::core_tree & core_tree, std::uint32_t stack_index);

std::regex
get_stream_name_regex(std::string const& protocol,
                      std::uint32_t stack_index,
                      std::string const& type);

template<typename Descriptor>
Descriptor
filter_stream_by_name(std::vector<Descriptor> descriptors,
                      std::string const & name)
{
    for (auto const& descriptor : descriptors)
        if (name == descriptor.get_properties().template get<hw::name>().v())
           return descriptor;
    std::ostringstream error;
    error << "Could not find stream " << name;
    throw std::runtime_error{error.str()};
}

template<typename Stream, typename Descriptors>
std::vector<Stream>
get_data_streams(Descriptors const& descriptors,
                 std::uint32_t stack_index,
                 std::string const& stack_type,
                 std::string const& type)
{
    auto regex = get_stream_name_regex(stack_type, stack_index, type);

    std::vector<Stream> streams;
    for (auto const& descriptor : descriptors)
    {
        auto const name = descriptor.get_properties().template get<hw::name>();
        if (! std::regex_match(name.value(), regex))
            continue;

        streams.emplace_back(descriptor);
    }

    if (streams.empty())
    {
        std::ostringstream error;
        error << stack_type << " stack " << stack_index
              << " has no associated " << type << " stream matching";
        throw std::runtime_error{error.str()};
    }

    return streams;
}

template<typename Source>
std::vector<Source>
get_sources(hw::accelerator & accelerator,
            std::uint32_t stack_index,
            std::string const& stack_type)
{
    auto const descriptors = accelerator.enumerate_a2c_streams();

    auto streams = get_data_streams<hw::a2c_stream>(descriptors,
                                                    stack_index,
                                                    stack_type,
                                                    "DATA");

    std::vector<Source> sources;
    std::move(streams.begin(), streams.end(), std::back_inserter(sources));
    return sources;
}

template<typename Sink>
std::vector<Sink>
get_sinks(hw::accelerator & accelerator,
          std::uint32_t stack_index,
          std::string const& stack_type)
{
    auto const descriptors = accelerator.enumerate_c2a_streams();

    auto streams = get_data_streams<hw::c2a_stream>(descriptors,
                                                    stack_index,
                                                    stack_type,
                                                    "DATA");

    std::vector<Sink> sinks;
    std::move(streams.begin(), streams.end(), std::back_inserter(sinks));
    return sinks;
}

hw::a2c_stream
get_tcp_events_source(hw::accelerator & accelerator,
                      std::uint32_t stack_index);

hw::c2a_stream
get_tcp_events_sink(hw::accelerator & accelerator,
                    std::uint8_t stack_index);

ENYX_CORES_NAMESPACE_END
