#include "DeviceDiscovery.hpp"

#include <vector>
#include <iterator>

#include <enyx/hw/properties.hpp>
#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/hw/mmio.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/data_stream/hw_source.hpp>
#include <enyx/cores/data_stream/hw_sink.hpp>
#include <enyx/cores/hardware_ids.hpp>

ENYX_CORES_NAMESPACE_BEGIN

static hw::filter
make_accelerator_filter(std::string const & arg)
{
    try {
        size_t end;
        uint32_t bus_id = std::stol(arg, &end, 0);
        if (end != arg.length())
            throw std::exception{};
        return hw::filter{hw::index{bus_id}};
    } catch (std::exception &) {
        return hw::filter{hw::name{arg}};
    }
}

hw::accelerator
create_accelerator(std::string const & accelerator_arg)
{
    // Construct a ethernet_filter that identify the accelerator
    hw::filter const filter{make_accelerator_filter(accelerator_arg)};

    // Find and instantiate the accelerator
    auto const accelerator_descriptors = hw::enumerate_accelerators(filter);
    if (accelerator_descriptors.size() != 1)
    {
        std::ostringstream error;
        error << "Can't find accelerator '"
              << accelerator_arg
              << "'";
        throw std::runtime_error{error.str()};
    }

    return hw::accelerator{accelerator_descriptors[0]};
}

hw::mmio
get_first_mmio(hw::accelerator & accelerator)
{
    // Construct a ethernet_filter that identify the first MMIO device
    hw::filter const filter{hw::index{0}};

    // Find and instantiate the first mmio from the accelerator
    auto const mmio_descriptors = accelerator.enumerate_mmios(filter);
    if (mmio_descriptors.size() != 1)
        throw std::runtime_error{"Excepting at least one mmio"};

    return hw::mmio{mmio_descriptors[0]};
}

hw::core
find_tcp_core(hw::core_tree & core_tree, std::uint32_t stack_index)
{
    std::uint8_t id = 0;
    for (const auto & core: core_tree) {
        auto descriptor = core.get_descriptor();
        if ((descriptor.hardware_id == enyx::hardware_ids::TCP_STD
             || descriptor.hardware_id == enyx::hardware_ids::TCP_ULL)
            && stack_index == id++)
                return core;
    }

    std::ostringstream error;
    error << "Expected at least " << (stack_index + 1) << " stack(s)";
    throw std::runtime_error{error.str()};
}

hw::core
find_udp_core(hw::core_tree & core_tree, std::uint32_t stack_index)
{
    std::uint8_t id = 0;
    for (const auto & core: core_tree) {
        auto descriptor = core.get_descriptor();
        if ((descriptor.hardware_id == enyx::hardware_ids::UDP_STD
             || descriptor.hardware_id == enyx::hardware_ids::UDP_ULL)
            && stack_index == id++)
                return core;
    }

    std::ostringstream error;
    error << "Expected at least " << (stack_index + 1) << " stack(s)";
    throw std::runtime_error{error.str()};
}

std::regex
get_stream_name_regex(std::string const& protocol,
                      std::uint32_t stack_index,
                      std::string const& type)
{
    // Build a regex, like "TCP0_CHAN\d+_DATA"
    std::ostringstream pattern;
    pattern << protocol << unsigned(stack_index) << R"(_CHAN\d+_)" << type;

    return std::regex{pattern.str()};
}

hw::a2c_stream
get_tcp_events_source(hw::accelerator & accelerator,
                      std::uint32_t stack_index)
{
    auto const descriptors = accelerator.enumerate_a2c_streams();

    auto streams = get_data_streams<hw::a2c_stream>(descriptors,
                                                    stack_index, "TCP",
                                                    "EMI");

    if (streams.size() != 1U)
    {
        std::ostringstream error;
        error << "Expected exactly 1 event stream associated with TCP stack "
              << stack_index << ", got " << streams.size();

        throw std::runtime_error{error.str()};
    }

    return std::move(streams[0]);
}

hw::c2a_stream
get_tcp_events_sink(hw::accelerator & accelerator,
                    std::uint8_t stack_index)
{
    auto const descriptors = accelerator.enumerate_c2a_streams();

    auto streams = get_data_streams<hw::c2a_stream>(descriptors,
                                                    stack_index, "TCP",
                                                    "EMI");

    if (streams.size() != 1U)
    {
        std::ostringstream error;
        error << "Expected exactly 1 event stream associated with TCP stack "
              << stack_index << ", got " << streams.size();

        throw std::runtime_error{error.str()};
    }

    return std::move(streams[0]);
}

ENYX_CORES_NAMESPACE_END
