#pragma once

#include <cstddef>
#include <cstdint>

#include <atomic>
#include <chrono>
#include <future>
#include <vector>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/core_tree.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/tcp/context.hpp>
#include <enyx/cores/tcp/tcp.hpp>

#include "ApplicationConfiguration.hpp"
#include "BandwidthThrottle.hpp"
#include "Buffer.hpp"
#include "CacheLine.hpp"
#include "TcpEventSource.hpp"
#include "TcpEventSource.hpp"
#include "TcpSink.hpp"
#include "TcpSource.hpp"
#include "Application.hpp"

ENYX_CORES_NAMESPACE_BEGIN

struct TcpRuntime final
{
    TcpRuntime(ApplicationConfiguration const & configuration,
               hw::accelerator & accelerator,
               hw::core_tree & core_tree);

    struct Session
    {
        SessionConfiguration const * configuration;

        enum State : std::uint8_t { UNUSED, CONFIGURED, ESTABLISHED };
        // This attribute is touched by event, tx & rx thread
        std::atomic<State> state;

        // This attribute is touched by the rx thread
        alignas(CACHE_LINE_SIZE) std::size_t received_bytes;

        // These attributes are touched by the tx thread
        alignas(CACHE_LINE_SIZE) std::size_t sent_bytes;

        std::size_t sink_backpressure;
        std::size_t tcp_backpressure;

        // This attribute is touched by both the event & tx thread
        std::atomic<std::uint32_t> tx_credit;
        BandwidthThrottle bandwidth_throttle;
        std::chrono::steady_clock::time_point begin;

        TxState get_tx_state();

        std::size_t get_payload_max_size() const;
    };

    using Sessions = std::vector<Session>;

    void
    on_state_change(std::uint32_t session_id,
                    tcp::context::state new_state,
                    tcp::context::state_info const* state_info);

    void
    on_credit_change(std::uint32_t session_id, std::uint32_t credit);

    void
    on_emi_error();

    void
    event_task();

    void
    create_monitoring_tasks(Tasks & tasks);

    void
    open_session(std::uint32_t session_id,
                 Session & session);

    void
    connect_session(std::uint32_t session_id,
                    Session & session);

    void
    listen_on_session(std::uint32_t session_id,
                      Session & session);

    bool
    is_tx_buffer_full(Session & session,
                      std::uint32_t size);

    void
    close_session(std::uint32_t session_id);

    template<typename Handle>
    int
    poll_source(std::size_t source_id, Handle & handle);

    std::vector<TcpSource> data_sources;
    std::vector<TcpSink> data_sinks;
    tcp::tcp stack;
    std::uint16_t session_count;
    Sessions sessions;

    tcp::context context;
    TcpEventSource events_source;
};

template<typename Handle>
inline int
TcpRuntime::poll_source(std::size_t source_id, Handle & handle)
{
    return data_sources[source_id].poll_once(handle);
}

ENYX_CORES_NAMESPACE_END
