#include "SessionConfiguration.hpp"

#include <stdexcept>
#include <iostream>
#include <sstream>

#include <enyx/cores/namespace.hpp>

ENYX_CORES_NAMESPACE_BEGIN

std::istream &
operator>>(std::istream & in, SessionConfiguration::Type & type)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        if (s == "client")
            type = SessionConfiguration::CLIENT;
        else if (s == "server")
            type = SessionConfiguration::SERVER;
        else if (s == "multicast")
            type = SessionConfiguration::MULTICAST;
        else
        {
            std::ostringstream error;
            error << "Unexpected type (" << s << ")";
            throw std::runtime_error(error.str());
        }
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const SessionConfiguration::Type & type)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    switch (type)
    {
    default:
    case SessionConfiguration::CLIENT:
        return out << "client";
    case SessionConfiguration::SERVER:
        return out << "server";
    case SessionConfiguration::MULTICAST:
        return out << "multicast";
    }
}

std::istream &
operator>>(std::istream & in, SessionConfiguration::Verify & verify)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        if (s == "none")
            verify = SessionConfiguration::NONE;
        else if (s == "first")
            verify = SessionConfiguration::FIRST;
        else if (s == "all")
            verify = SessionConfiguration::ALL;
        else
            throw std::runtime_error("Unexpected verification type");
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const SessionConfiguration::Verify & verify)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    switch (verify)
    {
    default:
    case SessionConfiguration::NONE:
        return out << "none";
    case SessionConfiguration::FIRST:
        return out << "first";
    case SessionConfiguration::ALL:
        return out << "all";
    }
}

std::ostream &
operator<<(std::ostream & out, const SessionConfiguration & configuration)
{
    std::ostream::sentry sentry(out);

    if (sentry)
    {
        out << "interface_id: " << configuration.interface_id << "\n";
        out << "type: " << configuration.type << "\n";
        out << "verify: " << configuration.verify << "\n";
        out << "endpoint: " << configuration.endpoint << "\n";
        out << "send_bandwidth: "
            << configuration.send_bandwidth << "/s\n";
        out << "mss: " << configuration.mss << "\n";
        out << "window: " << configuration.window << "\n";
        out << "retry_timeout: " << configuration.retry_timeout << "ms\n";
        out << "instant_ack: "
            << (configuration.instant_ack ? "enabled" : "disabled") << "\n";
        out << "size: " << configuration.size << "\n";
        out << "shutdown_policy: "
            << configuration.shutdown_policy << "\n";
        out << std::flush;
    }

    return out;
}

std::istream &
operator>>(std::istream & in, SessionConfiguration::ShutdownPolicy & policy)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        if (s == "wait_for_peer")
            policy = SessionConfiguration::WAIT_FOR_PEER;
        else if (s == "send_complete")
            policy = SessionConfiguration::SEND_COMPLETE;
        else if (s == "receive_complete")
            policy = SessionConfiguration::RECEIVE_COMPLETE;
        else
            throw std::runtime_error("Unexpected shutdown policy");
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const SessionConfiguration::ShutdownPolicy & policy)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    switch (policy)
    {
    default:
    case SessionConfiguration::WAIT_FOR_PEER:
        return out << "wait_for_peer";
    case SessionConfiguration::SEND_COMPLETE:
        return out << "send_complete";
    case SessionConfiguration::RECEIVE_COMPLETE:
        return out << "receive_complete";
    }
}

ENYX_CORES_NAMESPACE_END
