#include "ConfigurationParser.hpp"

#include <fstream>
#include <iostream>
#include <iterator>
#include <sstream>
#include <stdexcept>
#include <utility>

#include <boost/program_options.hpp>

#include "ApplicationConfiguration.hpp"
#include "SessionConfiguration.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace po = boost::program_options;

namespace {

void
parse_session_configurations(ApplicationConfiguration & app_configuration,
                             std::istream & in)
{
    SessionConfiguration c{};

    po::options_description session_required{"Required arguments"};
    session_required.add_options()
        ("type,t",
            po::value<SessionConfiguration::Type>(&c.type),
            "Act as a server or a client\n")
        ("endpoint,e",
            po::value<std::string>(&c.endpoint),
            "Listen on following port or connect to this address\n");

    po::options_description session_optional{"Optional arguments"};
    session_optional.add_options()
        ("net_interface-id,i",
            po::value<std::uint32_t>(&c.interface_id)
                ->default_value(0),
            "The stack net_interface id to use\n");

    po::options_description session_all{"SESSIONS OPTIONS"};
    session_all.add(session_required)
               .add(session_optional);

    // Iterate over each line of the session configuration file
    for (std::string line; std::getline(in, line); )
    {
        std::istringstream s{line};

        // Emulate a command line
        using tokenizer = std::istream_iterator<std::string>;
        std::vector<std::string> argv{"enyx-net-hw-tester"};
        argv.insert(argv.end(), tokenizer{s}, tokenizer{});

        auto session_parser = po::command_line_parser(argv)
                .options(session_all)
                .run();
        po::variables_map args;
        po::store(session_parser, args);
        po::notify(args);

        if (! args.count("type"))
            throw std::runtime_error{"--type is required"};

        if (! args.count("endpoint"))
            throw std::runtime_error{"--endpoint is required"};

        app_configuration.session_configurations.emplace_back(std::move(c));
    }
}

} // anonymous namespace

ApplicationConfiguration
parse_configuration(int argc, char ** argv)
{
    ApplicationConfiguration app_configuration{};

    std::string configuration_file;
    po::options_description cmd_optional{"Optional arguments"};
    cmd_optional.add_options()
        ("configuration-file,c",
            po::value<std::string>(&configuration_file)
                ->default_value("-"),
            "A file with one session configuration per line\n")
        ("accelerator-id,accelerator-index,a",
            po::value<std::string>(&app_configuration.accelerator)
                ->default_value("0"),
            "0-index or name of the accelerator to use\n")
        ("stack-index,s",
            po::value<std::uint32_t>(&app_configuration.stack_index)
                ->default_value(0),
            "0-index of the stack (UDP or TCP) to use\n")
        ("help,h",
            "Print the command lines arguments\n");

    po::options_description cmd_all{"COMMAND LINE OPTIONS"};
    cmd_all.add(cmd_optional);

    po::variables_map args;
    auto cmd_parser = po::command_line_parser(argc, argv)
            .options(cmd_all)
            .run();
    po::store(cmd_parser, args);
    po::notify(args);

    if (args.count("help"))
    {
        std::cout << cmd_all << std::endl;
        throw std::runtime_error{"help is requested"};
    }

    if (! args.count("configuration-file"))
        throw std::runtime_error{"--configuration-file argument is required"};

    if (configuration_file == "-")
    {
        parse_session_configurations(app_configuration, std::cin);
    }
    else
    {
        std::ifstream response_file{configuration_file};
        parse_session_configurations(app_configuration, response_file);
    }

    return app_configuration;
}

ENYX_CORES_NAMESPACE_END
