#include "Application.hpp"

#include <cerrno>

#include <future>
#include <sstream>
#include <stdexcept>
#include <vector>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/error.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/properties.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/net_interface/net_interface.hpp>

#include "InterfaceScope.hpp"
#include "ArpScope.hpp"
#include "IcmpScope.hpp"
#include "IgmpScope.hpp"
#include "FilterScope.hpp"
#include "EthernetFlowControlScope.hpp"

#include "../lib/DeviceDiscovery.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace {

char const USAGE[] =
"Usage: " BIN_NAME " [ OPTIONS ] interface\n"
"Usage: " BIN_NAME " [ OPTIONS ] arp\n"
"Usage: " BIN_NAME " [ OPTIONS ] icmp\n"
"Usage: " BIN_NAME " [ OPTIONS ] igmp\n"
"Usage: " BIN_NAME " [ OPTIONS ] ethernet_filter\n"
"Usage: " BIN_NAME " [ OPTIONS ] ethernet_flow_control\n"
"OPTIONS := { -a | --accelerator-id |\n"
"             -t | --stack-type STACK_TYPE |\n"
"             -s | --stack-index |\n"
"             -c | --color COLOR_MODE}\n"
"STACK_TYPE := { all | tcp_std | udp_std | tcp_ull | udp_ull | tcp | udp}\n"
"COLOR_MODE := { always | auto | never }\n";

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            hw::core & root,
            Application::StackType stack_type,
            std::uint32_t stack_index)
{
    if (i == e)
        throw std::runtime_error{"Missing SCOPE argument, "
                                 "try '" BIN_NAME " help'"};
    else if (is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "interface"))
        InterfaceScope::run(++i, e, root, stack_type, stack_index);
    else if (is_prefix(*i, "arp"))
        ArpScope::run(++i, e, root, stack_type, stack_index);
    else if (is_prefix(*i, "icmp"))
        IcmpScope::run(++i, e, root, stack_type, stack_index);
    else if (is_prefix(*i, "igmp"))
        IgmpScope::run(++i, e, root, stack_type, stack_index);
    else if (is_prefix(*i, "ethernet_filter"))
        FilterScope::run(++i, e, root, stack_type, stack_index);
    else if (is_prefix(*i, "ethernet_flow_control"))
        EthernetFlowControlScope::run(++i, e, root, stack_type, stack_index);
    else
    {
        std::ostringstream error;
        error <<  "Unknown SCOPE '" << *i << "', try '" BIN_NAME " help'";
        throw std::runtime_error{error.str()};
    }
}

} // anonymous namespace

namespace Application {

std::istream &
operator>>(std::istream & in, StackType & stack_type)
{
    std::string str;
    in >> str;

    if (str == "tcp_std")
        stack_type = StackType::TCP_STD;
    else if (str == "udp_std")
        stack_type = StackType::UDP_STD;
    else if (str == "tcp_ull")
        stack_type = StackType::TCP_ULL;
    else if (str == "udp_ull")
        stack_type = StackType::UDP_ULL;
    else if (str == "all")
        stack_type = StackType::ALL;
    else if (str == "tcp")
        stack_type = StackType::TCP;
    else if (str == "udp")
        stack_type = StackType::UDP;
    else
    {
        std::ostringstream error;
        error << "Unknown stack type '" << str << "'";
        throw std::runtime_error{error.str()};
    }

    return in;
}

void
run(CmdLineIterator i, CmdLineIterator e,
    std::string const & accelerator_arg,
    StackType stack_type,
    std::uint32_t stack_index)
{
    hw::accelerator accelerator{create_accelerator(accelerator_arg)};
    hw::mmio mmio{get_first_mmio(accelerator)};
    hw::core_tree core_tree{hw::enumerate_cores(mmio)};
    auto root = core_tree.get_root();

    parse_scope(i, e, root, stack_type, stack_index);
}

} // namespace Application

ENYX_CORES_NAMESPACE_END
