#include <boost/program_options.hpp>
namespace po = boost::program_options;

#include <fstream>
#include <iostream>
#include <iterator>

#include <stdio.h>

#include "common.hpp"
#include "runner.hpp"

flash_section
get_flash_section(const std::string & section)
{
    if (section == "pgm") {
        return flash_section::PGM;
    }
    if (section == "usr") {
        return flash_section::USR;
    }
    if (section == "prm") {
        return flash_section::PRM;
    }
    if (section == "otp") {
        return flash_section::OTP;
    }

    throw std::runtime_error("Unknown flash section " + section);
}

run_params
handle_options(po::variables_map args)
{
    run_params params;

    if (! args.count("accelerator")) {
        throw std::runtime_error("--accelerator option is mandatory");
    }

    params.accelerator = args["accelerator"].as<std::string>();

    if (args.count("read") == args.count("write")) {
        throw std::runtime_error("Either --read or --write must be specified");
    }

    if (args.count("section") == args.count("reg-addr")) {
        throw std::runtime_error(
                "Either --section or --reg-addr must be specified");
    }

    if (! args.count("size")) {
        throw std::runtime_error("--size must be specified");
    }
    params.size = args["size"].as<std::uint32_t>();

    if (args.count("section")) {
        if (! args.count("file")) {
            throw std::runtime_error(
                    "file must be provided when reading or writing memory");
        }
        params.file = args["file"].as<std::string>();

        if (args.count("offset")) {
            params.offset = args["offset"].as<std::uint32_t>();
        }

        params.section = args["section"].as<std::string>();

        if (args.count("section-id"))
            params.section_index = args["section-id"].as<std::uint32_t>();

        if (args.count("read")) {
            std::cout << "Reading "
                << params.size << " bytes "
                << "from memory section `" << params.section << "` ("
                << params.section_index << ") ";
            if (params.offset) {
                std::cout << "with an offset of " << params.offset << " bytes ";
            }
            std:: cout << "to file `" << params.file << "`" << std::endl;

            params.command = flash_command::READ_MEM;
        }

        if (args.count("write")) {
            std::cout << "Writing "
                << params.size << " bytes "
                << "to memory section `" << params.section << "` ("
                << params.section_index << ") ";
            if (params.offset) {
                std::cout << "with an offset of " << params.offset << " bytes ";
            }
            std:: cout << "from file `" << params.file << "`" << std::endl;

            params.command = flash_command::WRITE_MEM;
        }
    }

    if (args.count("reg-addr")) {
        params.reg_addr =
            std::stoul(args["reg-addr"].as<std::string>(), nullptr, 16);

        if (args.count("read")) {
            std::cout << "Reading register "
                << "0x" << std::hex << unsigned(params.reg_addr)
                << std::endl;

            params.command = flash_command::READ_REG;
        }

        if (args.count("write")) {
            if (! args.count("reg-value")) {
                throw std::runtime_error(
                        "reg-value must be provided when writing register");
            }
            params.reg_value =
                std::stoul(args["reg-value"].as<std::string>(), nullptr, 16);

            std::cout << "Writing register "
                << "0x" << std::hex << unsigned(params.reg_addr)
                << " with value "
                << std::hex << unsigned(params.reg_value)
                << std::endl;

            params.command = flash_command::WRITE_REG;
        }
    }

    return params;
}

po::variables_map
parse(int ac, char* av[])
{
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help,h", "produce help message")
        ("accelerator,a",
         po::value<std::string>(), "accelerator index or name")
        ("read,r", "read")
        ("write,w", "write")
        ("size,s", po::value<uint32_t>(), "size to read/write in bytes")
        ("timeout,t", po::value<uint64_t>(), "flash command timeout in ms")
        ;

    po::options_description mem_opts("Flash memory options");
    mem_opts.add_options()
        ("section", po::value<std::string>(), "flash section")
        ("section-id", po::value<uint32_t>(), "flash section index")
        ("offset", po::value<uint32_t>(), "offset from section start address")
        ("file,f", po::value<std::string>(), "input file or -")
        ;

    po::options_description reg_opts("Register options");
    reg_opts.add_options()
        ("reg-addr", po::value<std::string>(), "register address in hexadecimal notation")
        ("reg-value", po::value<std::string>(), "value to write in hexadecimal notation")
        ;

    desc.add(mem_opts).add(reg_opts);

    po::variables_map args;

    po::store(po::parse_command_line(ac, av, desc), args);
    po::notify(args);

    if (args.count("help")) {
        std::cout << desc << std::endl;
        exit(0);
    }

    return args;
}

int main(int ac, char* av[])
{
    auto args = parse(ac, av);

    auto params = handle_options(args);

    auto flash_runner = runner(params.accelerator);
    flash_runner.run(params);

    return 0;
}
