#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/udp/session.h>

#include <enyx/hw_c/core.h>

/**
 * The udp subsystem of a network stack.
 */
typedef struct enyx_udp enyx_udp;

/**
 * Construct the UDP module
 *
 * This module configures the following hw core:
 *
 * * udp_multi_stack (from UDP_STD 2.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible hw::core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_udp *
enyx_udp_create(enyx_hw_core * stack);

/**
 * Destroy a udp subsystem.
 *
 * @param udp The subsystem to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_udp_destroy(enyx_udp * udp);

/**
 * Get the name of a udp subsystem.
 *
 * @param udp The subsystem.
 * @param buffer Output buffer.
 * @param size size of buffer.
 *
 * @return The size of name on success, excluding the terminating null byte
 *         ('\0'). -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_udp_get_name(enyx_udp const * udp, char * buffer, size_t size);

/**
 * Get the UDP stack MTU.
 *
 * @param udp The subsystem to use.
 * @param mtu The stack MTU.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_mtu(enyx_udp const * udp, uint16_t * mtu);

/**
 * Is safe mode enabled on this udp subsystem.
 *
 * @note If the underlying hardware core is a udp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param udp The subsystem to use.
 * @return true if safe mode is enabled, false otherwise.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_udp_is_safe_mode_enabled(enyx_udp const * udp);

/**
 * Enable safe mode on this subsystem.
 *
 * @note If the underlying hardware core is a udp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param udp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_enable_safe_mode(enyx_udp * udp);

/**
 * Disable safe mode on this subsystem.
 *
 * @note If the underlying hardware core is a udp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param udp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_disable_safe_mode(enyx_udp * udp);

/**
 * Is checksum verification enabled on this udp subsystem.
 *
 * @note This feature is not supported by the udp_multi_stack hardware core.
 *
 * @param udp The subsystem to use.
 * @param enabled true if checksum verification is enabled, false otherwise.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_is_checksum_verification_enabled(enyx_udp const * udp,
                                          bool * enabled);

/**
 * Enable checksum verification on this subsystem.
 *
 * @note This feature is not supported by the udp_multi_stack hardware core.
 *
 * @param udp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_enable_checksum_verification(enyx_udp * udp);

/**
 * Disable checksum verification on this subsystem.
 *
 * @note This feature is not supported by the udp_multi_stack hardware core.
 *
 * @param udp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_disable_checksum_verification(enyx_udp * udp);

/**
 * Get the number of rx_sessions supported by a udp subsystem.
 *
 * @param udp The subsystem to use.
 * @return The number of rx_sessions.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t
enyx_udp_get_rx_session_count(enyx_udp const * udp);

/**
 * Get a rx_session object from a udp subsystem.
 *
 * @param udp The udp subsystem to use.
 * @param rx_session_id The id of the requested rx_session.
 * @param rx_session The rx_session object
 *
 * @note The @p rx_session needs the @p udp object to work, if you delete the
 * @p udp object, using the @p rx_session is undefined behavior.
 *
 * @return 0 on success, -1 on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_rx_session(enyx_udp const * udp, uint16_t rx_session_id,
                        enyx_udp_rx_session * rx_session);

/**
 * Get a rx_session that is currently closed.
 *
 * @note
 * The get_available_rx_session() is a specific function designed simply to
 * get the first UDP RX session that currently has the "CLOSED" status.
 * If you call `get_available_rx_session()` several times without opening the
 * session it will provide the same session every time.  If a session fails to
 * open and enters "CLOSED" status again, it will go back to the available pool
 * and may be obtained again.
 * If you need to get several sessions, we suggest using `get_rx_session`
 * instead.
 *
 * @param udp The subsystem to use.
 * @param rx_session The rx_session to be selected.
 * @return 0 if a rx_session was found. -1 on error (with @b errno
 *         set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_available_rx_session(enyx_udp const * udp,
                                  enyx_udp_rx_session * rx_session);

/**
 * Get the number of tx_sessions supported by a udp subsystem.
 *
 * @param udp The subsystem to use.
 * @return The number of tx_sessions.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t
enyx_udp_get_tx_session_count(enyx_udp const * udp);

/**
 * Get a tx_session object from a udp subsystem.
 *
 * @param udp The udp subsystem to use.
 * @param tx_session_id The id of the requested tx_session.
 * @param tx_session The tx_session object
 *
 * @note The @p tx_session needs the @p udp object to work, if you delete the
 * @p udp object, using the @p tx_session is undefined behavior.
 *
 * @return 0 on success, -1 on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_tx_session(enyx_udp const * udp, uint16_t tx_session_id,
                        enyx_udp_tx_session * tx_session);

/**
 * Get a tx_session that is currently closed.
 *
 * @note
 * The get_available_tx_session() is a specific function designed simply to
 * get the first UDP TX session that currently has the "CLOSED" status.
 * If you call `get_available_tx_session()` several times without opening the
 * session it will provide the same session every time.  If a session fails to
 * open and enters "CLOSED" status again, it will go back to the available pool
 * and may be obtained again.
 * If you need to get several sessions, we suggest using `get_tx_session`
 * instead.
 *
 * @param udp The subsystem to use.
 * @param tx_session The tx_session to be selected.
 * @return 0 if a tx_session was found. -1 on error (with @b errno
 *         set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_available_tx_session(enyx_udp const * udp,
                                  enyx_udp_tx_session * tx_session);

/**
 * A structure representing the UDP stack's runtime statistics.
 */
typedef struct {
    /**
     * Number of packets with UDP checksum errors, for all sessions.
     *
     * @note This value is not provided by the udp_multi_stack hardware core.
     */
    uint32_t rx_checksum_errors;
    /// The maximum capacity of the RX FIFO in packets.
    uint8_t rx_fifo_maximum_packet_count;
} enyx_udp_statistics;

/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_get_statistics_(enyx_udp const * udp,
                         enyx_udp_statistics * statistics,
                         size_t statistics_size);
/// @endcond

/**
 * Get the UDP subsystem Hardware Statistics
 *
 * @param udp The udp subsystem
 * @param statistics The statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_udp_get_statistics(udp, statistics) \
    enyx_udp_get_statistics_(udp, statistics, sizeof(*statistics))
