#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/types.h>

/**
 * Type representing a session.
 *
 * This can only be obtained from a udp object.
 */
typedef struct {
    /// Session ID
    uint16_t session_id;
    /// UDP subsystem this session is associated with
    struct enyx_udp * udp;
} enyx_udp_rx_session;

/**
 * UDP RX session open parameters.
 */
typedef struct {
    /// Interface to use when opening session.
    uint16_t interface_id;

    /// Source port to use for listening. Use 0 for random port.
    uint16_t source_port;
} enyx_udp_rx_session_parameters;

/**
 * Listen on session @p session. Port is specified via @p parameters port.
 *
 * @param session Session used for listening.
 * @param parameters Session parameters
 * @return 0 if session opening was successful. -1 on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_listen(enyx_udp_rx_session * session,
                           enyx_udp_rx_session_parameters const * parameters);

/**
 * Listen on @p multicast_group using @p session.
 * Port is specified via @p parameters port.
 *
 * @param session Session used for listening.
 * @param multicast_group Multicast group to use.
 * @param parameters Session parameters
 * @return 0 if session opening was successful. -1 on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_listen_multicast(enyx_udp_rx_session * session,
                                     enyx_ipv4_address const * multicast_group,
                                     enyx_udp_rx_session_parameters const * parameters);

/**
 * Close RX session @p session.
 *
 * @param session to close.
 * @return 0 if session was sucessfuly closed. -1 on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_close(enyx_udp_rx_session * session);

/**
 * Get the source port used of an open @p session.
 *
 * @param session An open session.
 * @param source_port The source port.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_get_source_port(enyx_udp_rx_session const * session,
                                    uint16_t * source_port);

/**
 * Possible state of sessions.
 */
typedef enum {
    /// Closed session.
    ENYX_UDP_RX_SESSION_STATE_CLOSED,
    /// Session is open.
    ENYX_UDP_RX_SESSION_STATE_LISTEN,
    /// Session is listening in multicast mode.
    ENYX_UDP_RX_SESSION_STATE_LISTEN_MULTICAST,
} enyx_udp_rx_session_state;

/**
 * @brief Get the state of a session.
 *
 * @param session The session.
 * @param state The state output.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_get_state(enyx_udp_rx_session const * session,
                              enyx_udp_rx_session_state * state);
/**
 * Type representing a session.
 *
 * This can only be obtained from a udp object.
 */
typedef struct {
    /// Session ID
    uint16_t session_id;
    /// UDP subsystem this session is associated with
    struct enyx_udp * udp;
} enyx_udp_tx_session;

/**
 * UDP TX session open parameters.
 */
typedef struct {
    /// Interface to use when opening session.
    uint16_t interface_id;

    /// Source port to use for connection. Use 0 for random port.
    uint16_t source_port;
} enyx_udp_tx_session_parameters;

/**
 * Connect to server @p peer_address @p port on @p session.
 *
 * @param session Session to connect.
 * @param peer_address Remote IPv4 address to connect to.
 * @param peer_port Remote port to connect to.
 * @param parameters Session parameters
 * @return 0 if session opening was successful. -1 on error (with @b errno set
 *         accordingly).
 *
 * @note This function does not wait for the sesssion to be established.
 *       Please check session status afterwards.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_connect(enyx_udp_tx_session * session,
                            enyx_ipv4_address const * peer_address,
                            uint16_t peer_port,
                            enyx_udp_tx_session_parameters const * parameters);

/**
 * Close session @p session.
 *
 * @param session to close.
 * @return 0 if session was sucessfuly closed. -1 on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_close(enyx_udp_tx_session * session);

/**
 * Get the source port used of an open @p session.
 *
 * @param session An open session.
 * @param source_port The source port.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_get_source_port(enyx_udp_tx_session const * session,
                                    uint16_t * source_port);

/**
 * A remote UDP endpoint.
 */
typedef enyx_ipv4_endpoint enyx_udp_endpoint;

/**
 * Get the peer (remote) @p address and @p port of an established @p session.
 *
 * @param session An established session.
 * @param peer The peer IPv4 and Port address.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_getpeername(enyx_udp_tx_session const * session,
                                enyx_udp_endpoint * peer);

/**
 * Possible state of sessions.
 */
typedef enum {
    /// Closed session.
    ENYX_UDP_TX_SESSION_STATE_CLOSED,
    /// Session is open.
    ENYX_UDP_TX_SESSION_STATE_OPEN,
} enyx_udp_tx_session_state;

/**
 * @brief Get the state of a session.
 *
 * @param session The session.
 * @param state The state output.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_get_state(enyx_udp_tx_session const * session,
                              enyx_udp_tx_session_state * state);

/**
 * A structure representing a RX session's runtime statistics.
 */
typedef struct {
    /// Number of user packets received
    uint32_t user_packet_count;
    /// Number of user bytes received
    uint32_t user_byte_count;
    /// Number of user packets dropped
    uint32_t user_drop_count;
} enyx_udp_rx_session_statistics;

/**
 * A structure representing a TX session's runtime statistics.
 */
typedef struct {
    /// Number of user packets transmitted
    uint32_t user_packet_count;
    /// Number of user bytes transmitted
    uint32_t user_byte_count;
    /// Number of user packets in error
    uint32_t user_error_count;
} enyx_udp_tx_session_statistics;

/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_rx_session_get_statistics_(enyx_udp_rx_session const * session,
                                    enyx_udp_rx_session_statistics * statistics,
                                    size_t statistics_size);
/// @endcond

/**
 * @brief Get the UDP subsystem RX session Hardware Statistics
 *
 * @param udp The udp subsystem
 * @param statistics The rx session statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_udp_rx_session_get_statistics(udp, statistics) \
    enyx_udp_rx_session_get_statistics_(udp, statistics, sizeof(*statistics))

/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_udp_tx_session_get_statistics_(enyx_udp_tx_session const * session,
                                    enyx_udp_tx_session_statistics * statistics,
                                    size_t statistics_size);
/// @endcond

/**
 * @brief Get the UDP subsystem TX session Hardware Statistics
 *
 * @param udp The udp subsystem
 * @param statistics The tx session statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_udp_tx_session_get_statistics(udp, statistics) \
    enyx_udp_tx_session_get_statistics_(udp, statistics, sizeof(*statistics))
